/**
 * Frontend JavaScript voor SEO Lead Pages Generator
 */

jQuery(document).ready(function($) {
    
    // Initialize lead form functionality
    initLeadForm();
    
    // Initialize template shortcodes
    initTemplateShortcodes();
    
    // Initialize page tracking
    initPageTracking();
    
    /**
     * Lead Form Initialisatie
     */
    function initLeadForm() {
        $('.lead-form').on('submit', function(e) {
            e.preventDefault();
            
            var $form = $(this);
            var $submitButton = $form.find('.submit-button');
            var $message = $form.siblings('.form-message');
            
            // Validate form
            if (!validateForm($form)) {
                return;
            }
            
            // Show loading state
            $submitButton.prop('disabled', true).text('Versturen...');
            $message.hide();
            
            // Collect form data
            var formData = {
                action: 'submit_lokaal_lead_fixed',
                nonce: seo_lead_pages_ajax.nonce,
                name: $form.find('[name="name"]').val(),
                email: $form.find('[name="email"]').val(),
                phone: $form.find('[name="phone"]').val(),
                message: $form.find('[name="message"]').val(),
                service: $form.find('[name="service"]').val(),
                location: $form.find('[name="location"]').val()
            };
            
            // Submit form via AJAX
            $.post(seo_lead_pages_ajax.ajax_url, formData, function(response) {
                if (response.success) {
                    showFormMessage($message, 'success', 'Bedankt! We nemen zo snel mogelijk contact met je op.');
                    $form[0].reset();
                    
                    // Track successful submission
                    trackLeadSubmission(formData);
                } else {
                    showFormMessage($message, 'error', 'Er is een fout opgetreden. Probeer het opnieuw.');
                }
            }).fail(function(xhr, status, error) {
                console.error('Form submission failed:', error);
                showFormMessage($message, 'error', 'Er is een fout opgetreden bij het versturen. Controleer je internetverbinding en probeer het opnieuw.');
            }).always(function() {
                $submitButton.prop('disabled', false).text('Versturen');
            });
        });
        
        // Real-time validation
        $('.lead-form input, .lead-form textarea').on('blur', function() {
            validateField($(this));
        });
        
        // Phone number formatting
        $('.lead-form input[name="phone"]').on('input', function() {
            formatPhoneNumber($(this));
        });
    }
    
    /**
     * Template Shortcodes Initialisatie
     */
    function initTemplateShortcodes() {
        // Process CTA button shortcodes
        $('[data-cta-button]').each(function() {
            var $button = $(this);
            var text = $button.data('cta-button');
            var style = $button.data('cta-style') || 'primary';
            
            $button.addClass('cta-button cta-button-' + style);
            $button.text(text);
            
            // Add click handler for smooth scrolling
            $button.on('click', function(e) {
                e.preventDefault();
                
                var target = $button.attr('href');
                if (target && target.startsWith('#')) {
                    var $targetElement = $(target);
                    if ($targetElement.length) {
                        $('html, body').animate({
                            scrollTop: $targetElement.offset().top - 100
                        }, 800);
                    }
                }
            });
        });
        
        // Process contact form shortcodes
        $('[data-contact-form]').each(function() {
            var $container = $(this);
            var title = $container.data('contact-form') || 'Neem contact op';
            var buttonText = $container.data('button-text') || 'Versturen';
            
            // Replace with actual form
            $container.html(generateContactForm(title, buttonText));
            
            // Reinitialize form functionality
            initLeadForm();
        });
    }
    
    /**
     * Page Tracking Initialisatie
     */
    function initPageTracking() {
        // Track page view
        if (typeof seo_lead_pages_ajax !== 'undefined') {
            trackPageView();
        }
        
        // Track scroll depth
        trackScrollDepth();
        
        // Track time on page
        trackTimeOnPage();
        
        // Track form interactions
        trackFormInteractions();
    }
    
    /**
     * Form Validation
     */
    function validateForm($form) {
        var isValid = true;
        
        // Validate required fields
        $form.find('[required]').each(function() {
            if (!validateField($(this))) {
                isValid = false;
            }
        });
        
        // Validate email format
        var $email = $form.find('[name="email"]');
        if ($email.val() && !isValidEmail($email.val())) {
            showFieldError($email, 'Voer een geldig email adres in');
            isValid = false;
        }
        
        return isValid;
    }
    
    function validateField($field) {
        var value = $field.val().trim();
        var isRequired = $field.prop('required');
        
        // Clear previous errors
        clearFieldError($field);
        
        // Check if required field is empty
        if (isRequired && !value) {
            showFieldError($field, 'Dit veld is verplicht');
            return false;
        }
        
        // Field is valid
        $field.addClass('valid');
        return true;
    }
    
    function showFieldError($field, message) {
        $field.addClass('error');
        
        var $error = $('<div class="field-error">' + message + '</div>');
        $field.after($error);
    }
    
    function clearFieldError($field) {
        $field.removeClass('error valid');
        $field.siblings('.field-error').remove();
    }
    
    function isValidEmail(email) {
        var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    /**
     * Phone Number Formatting
     */
    function formatPhoneNumber($input) {
        var value = $input.val().replace(/\D/g, '');
        
        if (value.length >= 10) {
            // Format as Dutch phone number
            if (value.startsWith('0')) {
                value = value.substring(1);
            }
            
            if (value.length === 9) {
                value = value.replace(/(\d{2})(\d{3})(\d{4})/, '$1 $2 $3');
            } else if (value.length === 10) {
                value = value.replace(/(\d{2})(\d{3})(\d{5})/, '$1 $2 $3');
            }
        }
        
        $input.val(value);
    }
    
    /**
     * Form Message Display
     */
    function showFormMessage($message, type, text) {
        $message.removeClass('success error')
               .addClass(type)
               .html(text)
               .fadeIn();
        
        // Auto-hide after 5 seconds
        setTimeout(function() {
            $message.fadeOut();
        }, 5000);
    }
    
    /**
     * Contact Form Generation
     */
    function generateContactForm(title, buttonText) {
        var phone = seo_lead_pages_ajax.phone_number || '';
        var email = seo_lead_pages_ajax.contact_email || '';
        
        var form = '<div class="contact-form-section">';
        form += '<h3>' + title + '</h3>';
        
        if (phone || email) {
            form += '<div class="contact-info">';
            if (phone) {
                form += '<p><strong>Telefoon:</strong> <a href="tel:' + phone + '">' + phone + '</a></p>';
            }
            if (email) {
                form += '<p><strong>Email:</strong> <a href="mailto:' + email + '">' + email + '</a></p>';
            }
            form += '</div>';
        }
        
        form += '<form class="lead-form" method="post">';
        form += '<div class="form-row"><label for="lead-name">Naam *</label><input type="text" id="lead-name" name="name" required></div>';
        form += '<div class="form-row"><label for="lead-email">Email *</label><input type="email" id="lead-email" name="email" required></div>';
        form += '<div class="form-row"><label for="lead-phone">Telefoon</label><input type="tel" id="lead-phone" name="phone"></div>';
        form += '<div class="form-row"><label for="lead-message">Bericht</label><textarea id="lead-message" name="message" rows="4"></textarea></div>';
        form += '<div class="form-row"><button type="submit" class="submit-button">' + buttonText + '</button></div>';
        form += '<input type="hidden" name="service" value="' + (seo_lead_pages_ajax.service || '') + '">';
        form += '<input type="hidden" name="location" value="' + (seo_lead_pages_ajax.location || '') + '">';
        form += '<input type="hidden" name="action" value="submit_lokaal_lead_fixed">';
        form += '<input type="hidden" name="nonce" value="' + seo_lead_pages_ajax.nonce + '">';
        form += '</form>';
        form += '<div class="form-message" style="display: none;"></div>';
        form += '</div>';
        
        return form;
    }
    
    /**
     * Analytics Tracking
     */
    function trackPageView() {
        if (typeof gtag !== 'undefined') {
            gtag('event', 'page_view', {
                'page_title': document.title,
                'page_location': window.location.href
            });
        }
        
        // Send to our analytics endpoint
        $.post(seo_lead_pages_ajax.ajax_url, {
            action: 'track_page_view',
            nonce: seo_lead_pages_ajax.nonce,
            page_id: seo_lead_pages_ajax.page_id || '',
            page_url: window.location.href
        });
    }
    
    function trackLeadSubmission(formData) {
        if (typeof gtag !== 'undefined') {
            gtag('event', 'lead_submission', {
                'event_category': 'lead_generation',
                'event_label': formData.service + ' - ' + formData.location,
                'value': 1
            });
        }
        
        // Send to our analytics endpoint
        $.post(seo_lead_pages_ajax.ajax_url, {
            action: 'track_lead_submission',
            nonce: seo_lead_pages_ajax.nonce,
            service: formData.service,
            location: formData.location
        });
    }
    
    function trackScrollDepth() {
        var scrollDepth = 0;
        var maxScroll = $(document).height() - $(window).height();
        
        $(window).on('scroll', function() {
            var currentScroll = $(window).scrollTop();
            var newScrollDepth = Math.round((currentScroll / maxScroll) * 100);
            
            if (newScrollDepth > scrollDepth && newScrollDepth % 25 === 0) {
                scrollDepth = newScrollDepth;
                
                if (typeof gtag !== 'undefined') {
                    gtag('event', 'scroll_depth', {
                        'event_category': 'engagement',
                        'event_label': scrollDepth + '%',
                        'value': scrollDepth
                    });
                }
            }
        });
    }
    
    function trackTimeOnPage() {
        var startTime = Date.now();
        
        $(window).on('beforeunload', function() {
            var timeOnPage = Math.round((Date.now() - startTime) / 1000);
            
            if (typeof gtag !== 'undefined') {
                gtag('event', 'time_on_page', {
                    'event_category': 'engagement',
                    'event_label': 'seconds',
                    'value': timeOnPage
                });
            }
        });
    }
    
    function trackFormInteractions() {
        $('.lead-form input, .lead-form textarea').on('focus', function() {
            if (typeof gtag !== 'undefined') {
                gtag('event', 'form_interaction', {
                    'event_category': 'lead_generation',
                    'event_label': 'form_focus'
                });
            }
        });
        
        $('.lead-form').on('submit', function() {
            if (typeof gtag !== 'undefined') {
                gtag('event', 'form_submit', {
                    'event_category': 'lead_generation',
                    'event_label': 'form_submit'
                });
            }
        });
    }
    
    /**
     * Utility Functions
     */
    
    // Debounce function for performance
    function debounce(func, wait) {
        var timeout;
        return function executedFunction() {
            var later = function() {
                clearTimeout(timeout);
                func();
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    // Throttle function for scroll events
    function throttle(func, limit) {
        var inThrottle;
        return function() {
            var args = arguments;
            var context = this;
            if (!inThrottle) {
                func.apply(context, args);
                inThrottle = true;
                setTimeout(function() {
                    inThrottle = false;
                }, limit);
            }
        };
    }
    
    // Smooth scroll to element
    function smoothScrollTo(element, offset) {
        var $element = $(element);
        if ($element.length) {
            $('html, body').animate({
                scrollTop: $element.offset().top - (offset || 0)
            }, 800);
        }
    }
    
    // Check if element is in viewport
    function isInViewport(element) {
        var rect = element.getBoundingClientRect();
        return (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
            rect.right <= (window.innerWidth || document.documentElement.clientWidth)
        );
    }
    
    // Lazy load images
    function initLazyLoading() {
        var images = document.querySelectorAll('img[data-src]');
        
        if ('IntersectionObserver' in window) {
            var imageObserver = new IntersectionObserver(function(entries, observer) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        var img = entry.target;
                        img.src = img.dataset.src;
                        img.classList.remove('lazy');
                        imageObserver.unobserve(img);
                    }
                });
            });
            
            images.forEach(function(img) {
                imageObserver.observe(img);
            });
        } else {
            // Fallback for older browsers
            images.forEach(function(img) {
                img.src = img.dataset.src;
            });
        }
    }
    
    // Initialize lazy loading
    initLazyLoading();
    
    // Performance monitoring
    if ('performance' in window) {
        window.addEventListener('load', function() {
            setTimeout(function() {
                var perfData = performance.getEntriesByType('navigation')[0];
                var loadTime = perfData.loadEventEnd - perfData.loadEventStart;
                
                if (typeof gtag !== 'undefined') {
                    gtag('event', 'page_load_time', {
                        'event_category': 'performance',
                        'event_label': 'load_time',
                        'value': Math.round(loadTime)
                    });
                }
            }, 0);
        });
    }
    
    // Error tracking
    window.addEventListener('error', function(e) {
        if (typeof gtag !== 'undefined') {
            gtag('event', 'javascript_error', {
                'event_category': 'error',
                'event_label': e.message,
                'value': 1
            });
        }
    });
    
    // Unhandled promise rejection tracking
    window.addEventListener('unhandledrejection', function(e) {
        if (typeof gtag !== 'undefined') {
            gtag('event', 'promise_rejection', {
                'event_category': 'error',
                'event_label': e.reason,
                'value': 1
            });
        }
    });
    
});
