/**
 * Admin JavaScript voor SEO Lead Pages Generator
 */

jQuery(document).ready(function($) {
    
    // Dashboard Charts
    if ($('#dashboard-chart').length) {
        initDashboardChart();
    }
    
    // Analytics Charts
    if ($('#analytics-chart').length) {
        initAnalyticsChart();
    }
    
    // Template Maker
    if ($('#template-maker').length) {
        initTemplateMaker();
    }
    
    // Locations Manager
    if ($('#locations-manager').length) {
        initLocationsManager();
    }
    
    // Generator
    if ($('#generator').length) {
        initGenerator();
    }
    
    // Lead Management
    if ($('#leads-management').length) {
        initLeadManagement();
    }
    
    // Import Locations
    if ($('#import-locations').length) {
        initImportLocations();
    }
    
    // SEO Overview
    if ($('#seo-overview').length) {
        initSEOOverview();
    }
    
    // Location Pages
    if ($('#location-pages').length) {
        initLocationPages();
    }
    
    // Bulk Operations
    if ($('#bulk-operations').length) {
        initBulkOperations();
    }
    
    // Sitemap Generation
    if ($('#sitemap-generation').length) {
        initSitemapGeneration();
    }
    
    // Competitor Analysis
    if ($('.competitor-analysis-results').length || $('.benchmark-results').length || $('button:contains("Analyze Competitors")').length) {
        initCompetitorAnalysis();
    }
    
    // Location Management
    if ($('#location-management').length) {
        initLocationManagement();
    }
    
    /**
     * Dashboard Chart Initialisatie
     */
    function initDashboardChart() {
        var ctx = document.getElementById('dashboard-chart').getContext('2d');
        
        $.post(ajaxurl, {
            action: 'get_analytics_data',
            nonce: seo_lead_pages_ajax.nonce,
            data_type: 'leads_by_month'
        }, function(response) {
            if (response.success) {
                var data = response.data;
                var labels = Object.keys(data);
                var values = Object.values(data);
                
                new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: labels,
                        datasets: [{
                            label: 'Leads per Maand',
                            data: values,
                            borderColor: '#667eea',
                            backgroundColor: 'rgba(102, 126, 234, 0.1)',
                            tension: 0.4,
                            fill: true
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: false
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: {
                                    stepSize: 1
                                }
                            }
                        }
                    }
                });
            }
        });
    }
    
    /**
     * Analytics Chart Initialisatie
     */
    function initAnalyticsChart() {
        var ctx = document.getElementById('analytics-chart').getContext('2d');
        
        $.post(ajaxurl, {
            action: 'get_analytics_data',
            nonce: seo_lead_pages_ajax.nonce,
            data_type: 'page_performance'
        }, function(response) {
            if (response.success) {
                var data = response.data;
                var labels = data.map(function(item) {
                    return item.title.substring(0, 20) + '...';
                });
                var views = data.map(function(item) {
                    return item.views || 0;
                });
                var leads = data.map(function(item) {
                    return item.leads || 0;
                });
                
                new Chart(ctx, {
                    type: 'bar',
                    data: {
                        labels: labels,
                        datasets: [{
                            label: 'Page Views',
                            data: views,
                            backgroundColor: 'rgba(102, 126, 234, 0.8)',
                            borderColor: '#667eea',
                            borderWidth: 1
                        }, {
                            label: 'Leads',
                            data: leads,
                            backgroundColor: 'rgba(255, 99, 132, 0.8)',
                            borderColor: '#ff6384',
                            borderWidth: 1
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: true
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
            }
        });
    }
    
    /**
     * Bulk Operations Initialisatie
     */
    function initBulkOperations() {
        // Bulk generate pages
        $('#bulk-generate-pages').on('click', function() {
            var service = $('#bulk-service-select').val();
            var locations = $('#bulk-locations-select').val();
            var template = $('#bulk-template-select').val();
            
            if (!service || !locations || locations.length === 0) {
                alert('Selecteer een dienst en locaties');
                return;
            }
            
            if (!confirm('Weet je zeker dat je ' + locations.length + ' pagina\'s wilt genereren?')) {
                return;
            }
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Genereren...');
            
            $.post(ajaxurl, {
                action: 'bulk_generate_pages',
                nonce: seo_lead_pages_ajax.nonce,
                service: service,
                locations: locations,
                template: template
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    
                    // Show results
                    var resultsHtml = '<div class="bulk-results">';
                    resultsHtml += '<h4>Resultaten:</h4>';
                    resultsHtml += '<p>Succesvol: ' + response.data.success_count + '</p>';
                    resultsHtml += '<p>Fouten: ' + response.data.error_count + '</p>';
                    
                    if (response.data.results.length > 0) {
                        resultsHtml += '<div class="results-list">';
                        response.data.results.forEach(function(result) {
                            var statusClass = result.status === 'success' ? 'success' : 'error';
                            resultsHtml += '<div class="result-item ' + statusClass + '">';
                            resultsHtml += '<strong>' + result.location + '</strong>: ' + result.message;
                            if (result.page_url) {
                                resultsHtml += ' <a href="' + result.page_url + '" target="_blank">Bekijk</a>';
                            }
                            resultsHtml += '</div>';
                        });
                        resultsHtml += '</div>';
                    }
                    
                    resultsHtml += '</div>';
                    
                    $('#bulk-results').html(resultsHtml).show();
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
        
        // Bulk SEO optimize
        $('#bulk-seo-optimize').on('click', function() {
            var selectedPages = $('input[name="page_ids[]"]:checked').map(function() {
                return $(this).val();
            }).get();
            
            if (selectedPages.length === 0) {
                alert('Selecteer pagina\'s om te optimaliseren');
                return;
            }
            
            if (!confirm('Weet je zeker dat je ' + selectedPages.length + ' pagina\'s wilt SEO optimaliseren?')) {
                return;
            }
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Optimaliseren...');
            
            $.post(ajaxurl, {
                action: 'bulk_seo_optimize',
                nonce: seo_lead_pages_ajax.nonce,
                page_ids: selectedPages
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
        
        // Bulk update content
        $('#bulk-update-content').on('click', function() {
            var selectedPages = $('input[name="page_ids[]"]:checked').map(function() {
                return $(this).val();
            }).get();
            
            var updateType = $('#bulk-update-type').val();
            
            if (selectedPages.length === 0) {
                alert('Selecteer pagina\'s om bij te werken');
                return;
            }
            
            if (!updateType) {
                alert('Selecteer een update type');
                return;
            }
            
            if (!confirm('Weet je zeker dat je ' + selectedPages.length + ' pagina\'s wilt bijwerken?')) {
                return;
            }
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Bijwerken...');
            
            $.post(ajaxurl, {
                action: 'bulk_update_content',
                nonce: seo_lead_pages_ajax.nonce,
                page_ids: selectedPages,
                update_type: updateType
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
    }
    
    /**
     * Sitemap Generation Initialisatie
     */
    function initSitemapGeneration() {
        $('#generate-sitemap').on('click', function() {
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Genereren...');
            
            $.post(ajaxurl, {
                action: 'generate_sitemap',
                nonce: seo_lead_pages_ajax.nonce
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    
                    // Show sitemap info
                    var sitemapHtml = '<div class="sitemap-info">';
                    sitemapHtml += '<h4>Sitemap gegenereerd:</h4>';
                    sitemapHtml += '<p><strong>URL:</strong> <a href="' + response.data.sitemap_url + '" target="_blank">' + response.data.sitemap_url + '</a></p>';
                    sitemapHtml += '<p><strong>Pagina\'s:</strong> ' + response.data.pages_count + '</p>';
                    sitemapHtml += '</div>';
                    
                    $('#sitemap-info').html(sitemapHtml).show();
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
    }
    
    /**
     * Import Locations Initialisatie
     */
    function initImportLocations() {
        // File upload change
        $('#import-file').on('change', function() {
            var fileName = this.files[0]?.name;
            if (fileName) {
                $('#file-name').text(fileName);
                $('#import-progress').show();
            }
        });
        
        // Import start
        $('#start-import').on('click', function() {
            var file = $('#import-file')[0].files[0];
            if (!file) {
                alert('Selecteer eerst een bestand');
                return;
            }
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Importeren...');
            
            // Simulate import process
            var progress = 0;
            var interval = setInterval(function() {
                progress += 10;
                $('.progress-fill').css('width', progress + '%');
                
                if (progress >= 100) {
                    clearInterval(interval);
                    $button.text('Import Voltooid!').addClass('button-primary');
                    setTimeout(function() {
                        $button.text(originalText).removeClass('button-primary');
                        $('#import-progress').hide();
                    }, 2000);
                }
            }, 200);
        });
    }
    
    /**
     * SEO Overview Initialisatie
     */
    function initSEOOverview() {
        // Refresh SEO metrics
        $('#refresh-seo').on('click', function() {
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Vernieuwen...');
            
            setTimeout(function() {
                $('.seo-metric .metric-value').each(function() {
                    var $metric = $(this);
                    var currentValue = parseInt($metric.text());
                    var newValue = currentValue + Math.floor(Math.random() * 10);
                    $metric.text(newValue);
                });
                
                setTimeout(function() {
                    $button.prop('disabled', false).text(originalText);
                    showNotification('SEO metrics vernieuwd', 'success');
                }, 500);
            }, 1000);
        });
        
        // Generate sitemap
        $('#generate-sitemap').on('click', function() {
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Genereren...');
            
            $.post(ajaxurl, {
                action: 'generate_sitemap',
                nonce: seo_lead_pages_ajax.nonce
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    
                    // Show sitemap info
                    var sitemapHtml = '<div class="sitemap-info">';
                    sitemapHtml += '<h4>Sitemap gegenereerd:</h4>';
                    sitemapHtml += '<p><strong>URL:</strong> <a href="' + response.data.sitemap_url + '" target="_blank">' + response.data.sitemap_url + '</a></p>';
                    sitemapHtml += '<p><strong>Pagina\'s:</strong> ' + response.data.pages_count + '</p>';
                    sitemapHtml += '</div>';
                    
                    $('#sitemap-info').html(sitemapHtml).show();
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
        
        // Check SEO Issues
        $('#check-seo-issues').on('click', function() {
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Checking...');
            
            $.post(ajaxurl, {
                action: 'check_seo_issues',
                nonce: seo_lead_pages_ajax.nonce
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    
                    // Show SEO issues results
                    var issuesHtml = '<div class="seo-issues-results" style="margin-top: 20px;">';
                    issuesHtml += '<h4>SEO Issues Found:</h4>';
                    
                    if (response.data.issues.length > 0) {
                        issuesHtml += '<ul style="margin: 10px 0; padding-left: 20px;">';
                        response.data.issues.forEach(function(issue) {
                            var severityClass = issue.severity === 'high' ? 'error' : (issue.severity === 'medium' ? 'warning' : 'info');
                            issuesHtml += '<li class="' + severityClass + '">';
                            issuesHtml += '<strong>' + issue.title + ':</strong> ' + issue.description;
                            if (issue.solution) {
                                issuesHtml += '<br><em>Solution: ' + issue.solution + '</em>';
                            }
                            issuesHtml += '</li>';
                        });
                        issuesHtml += '</ul>';
                    } else {
                        issuesHtml += '<p style="color: green;">No SEO issues found! Your pages are well optimized.</p>';
                    }
                    
                    issuesHtml += '</div>';
                    
                    $('#seo-issues-results').html(issuesHtml).show();
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
        
        // Run SEO Scan
        $('#run-seo-scan').on('click', function() {
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Scanning...');
            
            $.post(ajaxurl, {
                action: 'run_seo_scan',
                nonce: seo_lead_pages_ajax.nonce
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    
                    // Show SEO scan results
                    var scanHtml = '<div class="seo-scan-results" style="margin-top: 20px;">';
                    scanHtml += '<h4>SEO Scan Results:</h4>';
                    scanHtml += '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin: 15px 0;">';
                    scanHtml += '<div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">';
                    scanHtml += '<strong>Overall Score:</strong> ' + response.data.overall_score + '/100';
                    scanHtml += '</div>';
                    scanHtml += '<div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">';
                    scanHtml += '<strong>Pages Scanned:</strong> ' + response.data.pages_scanned;
                    scanHtml += '</div>';
                    scanHtml += '<div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">';
                    scanHtml += '<strong>Issues Found:</strong> ' + response.data.issues_found;
                    scanHtml += '</div>';
                    scanHtml += '</div>';
                    
                    if (response.data.recommendations.length > 0) {
                        scanHtml += '<h5>Recommendations:</h5>';
                        scanHtml += '<ul style="margin: 10px 0; padding-left: 20px;">';
                        response.data.recommendations.forEach(function(rec) {
                            scanHtml += '<li>' + rec + '</li>';
                        });
                        scanHtml += '</ul>';
                    }
                    
                    scanHtml += '</div>';
                    
                    $('#seo-scan-results').html(scanHtml).show();
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
    }
    
    /**
     * Competitor Analysis Initialisatie
     */
    function initCompetitorAnalysis() {
        // Analyze Competitors
        $('.button-primary').filter(function() {
            return $(this).text().includes('Analyze Competitors');
        }).on('click', function() {
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Analyzing...');
            
            $.post(ajaxurl, {
                action: 'analyze_competitors',
                nonce: seo_lead_pages_ajax.nonce
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    
                    // Show competitor analysis results
                    var analysisHtml = '<div class="competitor-analysis-results" style="margin-top: 20px;">';
                    analysisHtml += '<h4>Competitor Analysis Results:</h4>';
                    
                    if (response.data.competitors.length > 0) {
                        analysisHtml += '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 15px; margin: 15px 0;">';
                        
                        response.data.competitors.forEach(function(competitor) {
                            analysisHtml += '<div style="background: #f8f9fa; padding: 15px; border-radius: 8px; border: 1px solid #e9ecef;">';
                            analysisHtml += '<h5>' + competitor.name + '</h5>';
                            analysisHtml += '<p><strong>Domain Authority:</strong> ' + competitor.domain_authority + '</p>';
                            analysisHtml += '<p><strong>Backlinks:</strong> ' + competitor.backlinks + '</p>';
                            analysisHtml += '<p><strong>Keywords:</strong> ' + competitor.keywords + '</p>';
                            analysisHtml += '<p><strong>Traffic:</strong> ' + competitor.traffic + '</p>';
                            analysisHtml += '</div>';
                        });
                        
                        analysisHtml += '</div>';
                    } else {
                        analysisHtml += '<p>No competitors found for analysis.</p>';
                    }
                    
                    if (response.data.recommendations.length > 0) {
                        analysisHtml += '<h5>Recommendations:</h5>';
                        analysisHtml += '<ul style="margin: 10px 0; padding-left: 20px;">';
                        response.data.recommendations.forEach(function(rec) {
                            analysisHtml += '<li>' + rec + '</li>';
                        });
                        analysisHtml += '</ul>';
                    }
                    
                    analysisHtml += '</div>';
                    
                    $('.competitor-analysis-results').remove();
                    $button.closest('.dashboard-widget').after(analysisHtml);
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
        
        // View Benchmarks
        $('.button-primary').filter(function() {
            return $(this).text().includes('View Benchmarks');
        }).on('click', function() {
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Loading...');
            
            $.post(ajaxurl, {
                action: 'view_benchmarks',
                nonce: seo_lead_pages_ajax.nonce
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    
                    // Show benchmark results
                    var benchmarkHtml = '<div class="benchmark-results" style="margin-top: 20px;">';
                    benchmarkHtml += '<h4>Industry Benchmarks:</h4>';
                    
                    if (response.data.benchmarks.length > 0) {
                        benchmarkHtml += '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px; margin: 15px 0;">';
                        
                        response.data.benchmarks.forEach(function(benchmark) {
                            var scoreClass = benchmark.score >= 80 ? 'success' : (benchmark.score >= 60 ? 'warning' : 'error');
                            benchmarkHtml += '<div style="background: #f8f9fa; padding: 15px; border-radius: 8px; border: 1px solid #e9ecef;">';
                            benchmarkHtml += '<h5>' + benchmark.metric + '</h5>';
                            benchmarkHtml += '<p><strong>Your Score:</strong> <span class="' + scoreClass + '">' + benchmark.score + '/100</span></p>';
                            benchmarkHtml += '<p><strong>Industry Average:</strong> ' + benchmark.industry_average + '/100</p>';
                            benchmarkHtml += '<p><strong>Status:</strong> ' + benchmark.status + '</p>';
                            benchmarkHtml += '</div>';
                        });
                        
                        benchmarkHtml += '</div>';
                    } else {
                        benchmarkHtml += '<p>No benchmark data available.</p>';
                    }
                    
                    if (response.data.opportunities.length > 0) {
                        benchmarkHtml += '<h5>Improvement Opportunities:</h5>';
                        benchmarkHtml += '<ul style="margin: 10px 0; padding-left: 20px;">';
                        response.data.opportunities.forEach(function(opp) {
                            benchmarkHtml += '<li>' + opp + '</li>';
                        });
                        benchmarkHtml += '</ul>';
                    }
                    
                    benchmarkHtml += '</div>';
                    
                    $('.benchmark-results').remove();
                    $button.closest('.dashboard-widget').after(benchmarkHtml);
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
    }
    
    /**
     * Location Pages Initialisatie
     */
    function initLocationPages() {
        // Page actions
        $('.edit-page').on('click', function() {
            var pageId = $(this).data('page-id');
            window.open('post.php?post=' + pageId + '&action=edit', '_blank');
        });
        
        $('.view-page').on('click', function() {
            var pageUrl = $(this).data('page-url');
            window.open(pageUrl, '_blank');
        });
        
        $('.delete-page').on('click', function(e) {
            e.preventDefault();
            
            if (!confirm('Weet je zeker dat je deze pagina wilt verwijderen? Dit kan niet ongedaan worden gemaakt.')) {
                return;
            }
            
            var pageId = $(this).data('page-id');
            var $card = $(this).closest('.page-card');
            
            // Simulate deletion
            $card.fadeOut();
        });
    }
    
    /**
     * Template Maker Initialisatie
     */
    function initTemplateMaker() {
        // Template preview
        $('#template-content').on('input', function() {
            var content = $(this).val();
            if (content.length > 200) {
                $('#template-preview').html(content.substring(0, 200) + '...');
            } else {
                $('#template-preview').html(content);
            }
        });
        
        // Template save
        $('#save-template').on('click', function() {
            var templateName = $('#template-name').val();
            var templateContent = $('#template-content').val();
            
            if (!templateName || !templateContent) {
                alert('Vul alle velden in');
                return;
            }
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Opslaan...');
            
            $.post(ajaxurl, {
                action: 'save_template',
                nonce: seo_lead_pages_ajax.nonce,
                template_name: templateName,
                template_content: templateContent
            }, function(response) {
                if (response.success) {
                    showNotification('Template opgeslagen!', 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    showNotification('Fout bij opslaan: ' + response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden bij het opslaan.', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
        
        // Template delete
        $('.delete-template').on('click', function(e) {
            e.preventDefault();
            
            if (!confirm('Weet je zeker dat je deze template wilt verwijderen?')) {
                return;
            }
            
            var templateName = $(this).data('template');
            var $link = $(this);
            
            $.post(ajaxurl, {
                action: 'delete_template',
                nonce: seo_lead_pages_ajax.nonce,
                template_name: templateName
            }, function(response) {
                if (response.success) {
                    $link.closest('.template-item').fadeOut();
                    showNotification('Template verwijderd', 'success');
                } else {
                    showNotification('Fout bij verwijderen: ' + response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden bij het verwijderen.', 'error');
            });
        });
    }
    
    /**
     * Locations Manager Initialisatie
     */
    function initLocationsManager() {
        // Update location details
        $('.update-location').on('click', function(e) {
            e.preventDefault();
            
            var locationId = $(this).data('location-id');
            var population = $('#population-' + locationId).val();
            var landmarks = $('#landmarks-' + locationId).val();
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Bijwerken...');
            
            $.post(ajaxurl, {
                action: 'update_location_details',
                nonce: seo_lead_pages_ajax.nonce,
                location_id: locationId,
                population: population,
                landmarks: landmarks
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
        
        // Location search
        $('#location-search').on('input', function() {
            var searchTerm = $(this).val().toLowerCase();
            
            $('.location-card').each(function() {
                var locationName = $(this).find('.location-name').text().toLowerCase();
                var provinceName = $(this).find('.location-province').text().toLowerCase();
                
                if (locationName.includes(searchTerm) || provinceName.includes(searchTerm)) {
                    $(this).show();
                } else {
                    $(this).hide();
                }
            });
        });
        
        // Clean duplicates
        $('#clean-duplicates').on('click', function() {
            if (!confirm('Weet je zeker dat je duplicaten wilt verwijderen?')) {
                return;
            }
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Verwijderen...');
            
            $.post(ajaxurl, {
                action: 'clean_duplicate_locations',
                nonce: seo_lead_pages_ajax.nonce
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
    }
    
    /**
     * Generator Initialisatie
     */
    function initGenerator() {
        // Service selection
        $('#service-select').on('change', function() {
            var service = $(this).val();
            if (service) {
                $('#location-select').prop('disabled', false);
            } else {
                $('#location-select').prop('disabled', true);
            }
        });
        
        // Location selection
        $('#location-select').on('change', function() {
            var location = $(this).val();
            if (location) {
                $('#generate-page').prop('disabled', false);
            } else {
                $('#generate-page').prop('disabled', true);
            }
        });
        
        // Generate page
        $('#generate-page').on('click', function() {
            var service = $('#service-select').val();
            var location = $('#location-select').val();
            var template = $('#template-select').val();
            var pageTitle = $('#page-title').val();
            var metaDescription = $('#meta-description').val();
            var focusKeyword = $('#focus-keyword').val();
            var includeReviews = $('#include-reviews').is(':checked');
            
            if (!service || !location) {
                alert('Selecteer een dienst en locatie');
                return;
            }
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Genereren...');
            
            $.post(ajaxurl, {
                action: 'generate_landing_page',
                nonce: seo_lead_pages_ajax.nonce,
                service: service,
                location: location,
                template: template,
                page_title: pageTitle,
                meta_description: metaDescription,
                focus_keyword: focusKeyword,
                include_reviews: includeReviews
            }, function(response) {
                if (response.success) {
                    $('#generated-content').html(response.data.html).show();
                    showNotification('Landingspagina succesvol gegenereerd!', 'success');
                } else {
                    showNotification('Fout bij genereren: ' + response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            }).always(function() {
                $button.prop('disabled', false).text(originalText);
            });
        });
        
        // Auto-fill page title and meta description
        $('#service-select, #location-select').on('change', function() {
            var service = $('#service-select').val();
            var location = $('#location-select').val();
            
            if (service && location) {
                if (!$('#page-title').val()) {
                    $('#page-title').val(service + ' in ' + location);
                }
                if (!$('#meta-description').val()) {
                    $('#meta-description').val('Professionele ' + service + ' diensten in ' + location + '. Neem contact op voor een gratis offerte!');
                }
                if (!$('#focus-keyword').val()) {
                    $('#focus-keyword').val(service + ' ' + location);
                }
            }
        });
    }
    
    /**
     * Lead Management Initialisatie
     */
    function initLeadManagement() {
        // Update lead status
        $('.update-lead-status').on('change', function() {
            var leadId = $(this).data('lead-id');
            var newStatus = $(this).val();
            
            $.post(ajaxurl, {
                action: 'update_lead_status',
                nonce: seo_lead_pages_ajax.nonce,
                lead_id: leadId,
                new_status: newStatus
            }, function(response) {
                if (response.success) {
                    showNotification(response.data.message, 'success');
                } else {
                    showNotification(response.data.message, 'error');
                }
            }).fail(function() {
                showNotification('Er is een fout opgetreden', 'error');
            });
        });
        
        // Export leads
        $('#export-leads').on('click', function() {
            var dateFrom = $('#date-from').val();
            var dateTo = $('#date-to').val();
            
            if (!dateFrom || !dateTo) {
                alert('Selecteer een datum bereik');
                return;
            }
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Exporteren...');
            
            // Create download link
            var downloadUrl = ajaxurl + '?action=export_leads&nonce=' + seo_lead_pages_ajax.nonce + '&date_from=' + dateFrom + '&date_to=' + dateTo;
            
            var link = document.createElement('a');
            link.href = downloadUrl;
            link.download = 'leads-' + dateFrom + '-to-' + dateTo + '.csv';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            setTimeout(function() {
                $button.prop('disabled', false).text(originalText);
                showNotification('Export gestart', 'success');
            }, 1000);
        });
    }
    
    /**
     * Utility Functions
     */
    
    // Format numbers with commas
    function formatNumber(num) {
        return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
    }
    
    // Show notification
    function showNotification(message, type) {
        var $notification = $('<div class="notice notice-' + type + '"><p>' + message + '</p></div>');
        $('.wrap h1').after($notification);
        
        setTimeout(function() {
            $notification.fadeOut();
        }, 3000);
    }
    
    // Confirm action
    function confirmAction(message) {
        return confirm(message);
    }
    
    // Loading state
    function setLoadingState($element, isLoading) {
        if (isLoading) {
            $element.addClass('loading');
        } else {
            $element.removeClass('loading');
        }
    }
    
    // Error handling
    function handleError(error, context) {
        console.error('Error in ' + context + ':', error);
        showNotification('Er is een fout opgetreden: ' + error.message, 'error');
    }
    
    // Success handling
    function handleSuccess(message) {
        showNotification(message, 'success');
    }
    
    // AJAX error handler
    $(document).ajaxError(function(event, xhr, settings, error) {
        console.error('AJAX Error:', error);
        showNotification('Er is een fout opgetreden bij het laden van de gegevens.', 'error');
    });
    
    // Global error handler
    window.addEventListener('error', function(e) {
        console.error('Global Error:', e.error);
        console.error('Error details:', {
            message: e.message,
            filename: e.filename,
            lineno: e.lineno,
            colno: e.colno,
            error: e.error
        });
        
        // Only show notification for critical errors, not for expected AJAX failures
        if (e.message && !e.message.includes('ajax') && !e.message.includes('XMLHttpRequest')) {
            showNotification('Er is een onverwachte fout opgetreden. Controleer de console voor details.', 'error');
        }
    });
    
    // Unhandled promise rejection handler
    window.addEventListener('unhandledrejection', function(e) {
        console.error('Unhandled Promise Rejection:', e.reason);
        showNotification('Er is een fout opgetreden bij het verwerken van de gegevens.', 'error');
    });
    
});
