<?php
/**
 * Shortcodes Class for SEO Lead Pages Generator
 * Handles all shortcode functionality
 * Compatible with both standalone and addon versions
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class SEO_Lead_Pages_Shortcodes {
    
    public function __construct() {
        add_shortcode('phone_number', array($this, 'phone_number_shortcode'));
        add_shortcode('contact_email', array($this, 'contact_email_shortcode'));
        add_shortcode('service_name', array($this, 'service_name_shortcode'));
        add_shortcode('location_name', array($this, 'location_name_shortcode'));
        add_shortcode('province_name', array($this, 'province_name_shortcode'));
        add_shortcode('population', array($this, 'population_shortcode'));
        add_shortcode('landmarks', array($this, 'landmarks_shortcode'));
        add_shortcode('contact_form', array($this, 'contact_form_shortcode'));
        add_shortcode('cta_button', array($this, 'cta_button_shortcode'));
    }
    
    /**
     * Phone number shortcode
     */
    public function phone_number_shortcode($atts) {
        $atts = shortcode_atts(array(
            'format' => 'display', // display, link, or both
            'class' => 'phone-number'
        ), $atts);
        
        // Check if helper function exists (for addon compatibility)
        if (function_exists('seo_lead_pages_get_phone_number')) {
            $phone_number = seo_lead_pages_get_phone_number();
        } else {
            $phone_number = get_option('seo_lead_pages_phone_number', '');
        }
        
        if (empty($phone_number)) {
            return '';
        }
        
        $output = '';
        
        if ($atts['format'] === 'link' || $atts['format'] === 'both') {
            $output .= '<a href="tel:' . esc_attr($phone_number) . '" class="' . esc_attr($atts['class']) . '-link">';
        }
        
        $output .= '<span class="' . esc_attr($atts['class']) . '">' . esc_html($phone_number) . '</span>';
        
        if ($atts['format'] === 'link' || $atts['format'] === 'both') {
            $output .= '</a>';
        }
        
        return $output;
    }
    
    /**
     * Contact email shortcode
     */
    public function contact_email_shortcode($atts) {
        $atts = shortcode_atts(array(
            'format' => 'display', // display, link, or both
            'class' => 'contact-email'
        ), $atts);
        
        // Check if helper function exists (for addon compatibility)
        if (function_exists('seo_lead_pages_get_contact_email')) {
            $contact_email = seo_lead_pages_get_contact_email();
        } else {
            $contact_email = get_option('seo_lead_pages_contact_email', get_option('admin_email'));
        }
        
        if (empty($contact_email)) {
            return '';
        }
        
        $output = '';
        
        if ($atts['format'] === 'link' || $atts['format'] === 'both') {
            $output .= '<a href="mailto:' . esc_attr($contact_email) . '" class="' . esc_attr($atts['class']) . '-link">';
        }
        
        $output .= '<span class="' . esc_attr($atts['class']) . '">' . esc_html($contact_email) . '</span>';
        
        if ($atts['format'] === 'link' || $atts['format'] === 'both') {
            $output .= '</a>';
        }
        
        return $output;
    }
    
    /**
     * Service name shortcode
     */
    public function service_name_shortcode($atts) {
        $atts = shortcode_atts(array(
            'class' => 'service-name',
            'fallback' => 'onze diensten'
        ), $atts);
        
        global $post;
        
        if (!$post) {
            return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($atts['fallback']) . '</span>';
        }
        
        $service = get_post_meta($post->ID, '_seo_lead_pages_service', true);
        
        if (empty($service)) {
            $service = $atts['fallback'];
        }
        
        return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($service) . '</span>';
    }
    
    /**
     * Location name shortcode
     */
    public function location_name_shortcode($atts) {
        $atts = shortcode_atts(array(
            'class' => 'location-name',
            'fallback' => 'deze locatie'
        ), $atts);
        
        global $post;
        
        if (!$post) {
            return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($atts['fallback']) . '</span>';
        }
        
        $location = get_post_meta($post->ID, '_seo_lead_pages_location', true);
        
        if (empty($location)) {
            $location = $atts['fallback'];
        }
        
        return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($location) . '</span>';
    }
    
    /**
     * Province name shortcode
     */
    public function province_name_shortcode($atts) {
        $atts = shortcode_atts(array(
            'class' => 'province-name',
            'fallback' => 'deze provincie'
        ), $atts);
        
        global $post;
        
        if (!$post) {
            return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($atts['fallback']) . '</span>';
        }
        
        $province = get_post_meta($post->ID, '_seo_lead_pages_province', true);
        
        if (empty($province)) {
            $province = $atts['fallback'];
        }
        
        return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($province) . '</span>';
    }
    
    /**
     * Population shortcode
     */
    public function population_shortcode($atts) {
        $atts = shortcode_atts(array(
            'class' => 'population',
            'format' => 'number', // number, formatted, or text
            'fallback' => 'veel inwoners'
        ), $atts);
        
        global $post;
        
        if (!$post) {
            return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($atts['fallback']) . '</span>';
        }
        
        $location = get_post_meta($post->ID, '_seo_lead_pages_location', true);
        
        if (empty($location)) {
            return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($atts['fallback']) . '</span>';
        }
        
        global $wpdb;
        $population = $wpdb->get_var($wpdb->prepare(
            "SELECT population FROM {$wpdb->prefix}seo_lead_pages_locations WHERE name = %s",
            $location
        ));
        
        if (empty($population)) {
            return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($atts['fallback']) . '</span>';
        }
        
        if ($atts['format'] === 'formatted') {
            $population = number_format($population, 0, ',', '.');
        } elseif ($atts['format'] === 'text') {
            $population = $population . ' inwoners';
        }
        
        return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($population) . '</span>';
    }
    
    /**
     * Landmarks shortcode
     */
    public function landmarks_shortcode($atts) {
        $atts = shortcode_atts(array(
            'class' => 'landmarks',
            'fallback' => 'veel bezienswaardigheden'
        ), $atts);
        
        global $post;
        
        if (!$post) {
            return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($atts['fallback']) . '</span>';
        }
        
        $location = get_post_meta($post->ID, '_seo_lead_pages_location', true);
        
        if (empty($location)) {
            return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($atts['fallback']) . '</span>';
        }
        
        global $wpdb;
        $landmarks = $wpdb->get_var($wpdb->prepare(
            "SELECT landmarks FROM {$wpdb->prefix}seo_lead_pages_locations WHERE name = %s",
            $location
        ));
        
        if (empty($landmarks)) {
            return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($atts['fallback']) . '</span>';
        }
        
        return '<span class="' . esc_attr($atts['class']) . '">' . esc_html($landmarks) . '</span>';
    }
    
    /**
     * Contact form shortcode
     */
    public function contact_form_shortcode($atts) {
        $atts = shortcode_atts(array(
            'title' => 'Contact us',
            'button_text' => 'Versturen',
            'class' => 'seo-lead-form'
        ), $atts);
        
        global $post;
        
        $service = get_post_meta($post->ID, '_seo_lead_pages_service', true);
        $location = get_post_meta($post->ID, '_seo_lead_pages_location', true);
        
        ob_start();
        ?>
        <div class="contact-form-section">
            <?php if (!empty($atts['title'])): ?>
            <h3><?php echo esc_html($atts['title']); ?></h3>
            <?php endif; ?>
            
            <form class="<?php echo esc_attr($atts['class']); ?>" method="post">
                <div class="form-row">
                    <label for="lead-name">Naam *</label>
                    <input type="text" id="lead-name" name="name" required>
                </div>
                
                <div class="form-row">
                    <label for="lead-email">Email *</label>
                    <input type="email" id="lead-email" name="email" required>
                </div>
                
                <div class="form-row">
                    <label for="lead-phone">Telefoon</label>
                    <input type="tel" id="lead-phone" name="phone">
                </div>
                
                <div class="form-row">
                    <label for="lead-message">Bericht</label>
                    <textarea id="lead-message" name="message" rows="4"></textarea>
                </div>
                
                <div class="form-row">
                    <button type="submit" class="submit-button"><?php echo esc_html($atts['button_text']); ?></button>
                </div>
                
                <input type="hidden" name="service" value="<?php echo esc_attr($service); ?>">
                <input type="hidden" name="location" value="<?php echo esc_attr($location); ?>">
                <input type="hidden" name="action" value="submit_lokaal_lead_fixed">
                <?php wp_nonce_field('seo_lead_pages_nonce', 'nonce'); ?>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * CTA button shortcode
     */
    public function cta_button_shortcode($atts) {
        $atts = shortcode_atts(array(
            'text' => 'Klik hier',
            'style' => 'primary', // primary, secondary, outline
            'link' => '#',
            'class' => 'cta-button'
        ), $atts);
        
        $button_class = $atts['class'] . ' ' . $atts['style'];
        
        return '<a href="' . esc_url($atts['link']) . '" class="' . esc_attr($button_class) . '">' . esc_html($atts['text']) . '</a>';
    }
}
?>
