<?php
/**
 * Frontend Class for SEO Lead Pages Generator
 * Handles frontend functionality and shortcode processing
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class SEO_Lead_Pages_Frontend {
    
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_head', array($this, 'add_meta_tags'));
        add_action('wp_footer', array($this, 'add_structured_data'));
        add_shortcode('seo_lead_form', array($this, 'lead_form_shortcode'));
    }
    
    public function enqueue_scripts() {
        wp_enqueue_script('jquery');
        
        wp_enqueue_style(
            'seo-lead-pages-frontend',
            plugin_dir_url(__FILE__) . '../css/frontend.css',
            array(),
            '1.0.0'
        );
        
        wp_enqueue_script('seo-lead-pages-frontend', 
            plugin_dir_url(__FILE__) . '../js/frontend.js', 
            array('jquery'), 
            '1.0.0', 
            true
        );
        
        wp_localize_script('seo-lead-pages-frontend', 'seo_lead_pages_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seo_lead_pages_nonce')
        ));
    }
    
    public function add_meta_tags() {
        // Add Open Graph tags for better social media sharing
        if (is_page()) {
            global $post;
            
            $og_image = get_option('seo_lead_pages_og_image', '');
            if ($og_image) {
                echo '<meta property="og:image" content="' . esc_url($og_image) . '" />' . "\n";
            }
            
            $phone = get_option('seo_lead_pages_phone_number', '');
            if ($phone) {
                echo '<meta property="og:phone_number" content="' . esc_attr($phone) . '" />' . "\n";
            }
        }
    }
    
    public function add_structured_data() {
        // Add JSON-LD structured data for better SEO
        if (is_page()) {
            global $post;
            
            $service = get_post_meta($post->ID, '_seo_lead_pages_service', true);
            $location = get_post_meta($post->ID, '_seo_lead_pages_location', true);
            $province = get_post_meta($post->ID, '_seo_lead_pages_province', true);
            
            if ($service && $location) {
                $structured_data = array(
                    '@context' => 'https://schema.org',
                    '@type' => 'LocalBusiness',
                    'name' => $service . ' in ' . $location,
                    'description' => 'Professionele ' . $service . ' diensten in ' . $location . ', ' . $province,
                    'address' => array(
                        '@type' => 'PostalAddress',
                        'addressLocality' => $location,
                        'addressRegion' => $province,
                        'addressCountry' => 'NL'
                    ),
                    'telephone' => get_option('seo_lead_pages_phone_number', ''),
                    'email' => get_option('seo_lead_pages_contact_email', ''),
                    'url' => get_permalink($post->ID),
                    'sameAs' => array(
                        'https://stormseo.nl'
                    )
                );
                
                echo '<script type="application/ld+json">' . wp_json_encode($structured_data) . '</script>' . "\n";
            }
        }
    }
    
    public function lead_form_shortcode($atts) {
        $atts = shortcode_atts(array(
            'title' => '',
            'button_text' => '',
            'service' => '',
            'location' => '',
            'phone' => '',
            'email' => '',
            'website' => ''
        ), $atts);
        
        // Get form text settings
        $form_title = get_option('seo_lead_pages_form_title', __('Free SEO Advice for', 'geo-local-seo-generator'));
        $contact_label = get_option('seo_lead_pages_contact_label', __('Direct contact:', 'geo-local-seo-generator'));
        $name_label = get_option('seo_lead_pages_name_label', __('Name', 'geo-local-seo-generator'));
        $email_label = get_option('seo_lead_pages_email_label', __('Email address', 'geo-local-seo-generator'));
        $phone_label = get_option('seo_lead_pages_phone_label', __('Phone number', 'geo-local-seo-generator'));
        $company_label = get_option('seo_lead_pages_company_label', __('Company name', 'geo-local-seo-generator'));
        $message_label = get_option('seo_lead_pages_message_label', __('Message', 'geo-local-seo-generator'));
        $message_placeholder = get_option('seo_lead_pages_message_placeholder', __('Tell us about your SEO goals and challenges...', 'geo-local-seo-generator'));
        $submit_button = get_option('seo_lead_pages_submit_button', __('Request Free SEO Advice', 'geo-local-seo-generator'));
        
        // Override with shortcode attributes if provided
        if (!empty($atts['title'])) {
            $form_title = $atts['title'];
        }
        if (!empty($atts['button_text'])) {
            $submit_button = $atts['button_text'];
        }
        
        // Get contact display options
        $show_phone = get_option('seo_lead_pages_show_phone', 1);
        $show_email = get_option('seo_lead_pages_show_email', 1);
        $show_website = get_option('seo_lead_pages_show_website', 1);
        
        // Get form styling options
        $form_background_color = get_option('seo_lead_pages_form_background_color', '#ffffff');
        $form_border_color = get_option('seo_lead_pages_form_border_color', '#e1e5e9');
        $form_text_color = get_option('seo_lead_pages_form_text_color', '#333333');
        $form_title_color = get_option('seo_lead_pages_form_title_color', '#0073aa');
        $input_background_color = get_option('seo_lead_pages_input_background_color', '#ffffff');
        $input_border_color = get_option('seo_lead_pages_input_border_color', '#ddd');
        $input_text_color = get_option('seo_lead_pages_input_text_color', '#333333');
        $button_background_color = get_option('seo_lead_pages_button_background_color', '#0073aa');
        $button_text_color = get_option('seo_lead_pages_button_text_color', '#ffffff');
        $button_hover_color = get_option('seo_lead_pages_button_hover_color', '#005a87');
        $form_border_radius = get_option('seo_lead_pages_form_border_radius', 8);
        $form_padding = get_option('seo_lead_pages_form_padding', 30);
        $form_width = get_option('seo_lead_pages_form_width', '100%');
        $form_alignment = get_option('seo_lead_pages_form_alignment', 'left');
        
        // Debug: Log the styling options
        error_log('GEO Local SEO Debug - Frontend styling options:');
        error_log('Background: ' . $form_background_color);
        error_log('Button: ' . $button_background_color);
        error_log('Title: ' . $form_title_color);
        
        // Build contact info section
        $contact_info = '';
        if ($show_phone || $show_email || $show_website) {
            $contact_info = '<div class="contact-info">';
            $contact_info .= '<p><strong>' . esc_html($contact_label) . '</strong></p>';
            
            if ($show_phone && !empty($phone)) {
                $contact_info .= '<p>📞 <a href="tel:' . esc_attr($phone) . '">' . esc_html($phone) . '</a></p>';
            }
            
            if ($show_email && !empty($email)) {
                $contact_info .= '<p>📧 <a href="mailto:' . esc_attr($email) . '">' . esc_html($email) . '</a></p>';
            }
            
            if ($show_website && !empty($website)) {
                $contact_info .= '<p>🌐 <a href="' . esc_url($website) . '" target="_blank">' . esc_html($website) . '</a></p>';
            }
            
            $contact_info .= '</div>';
        }
        
        // Default values if not provided
        $service = !empty($atts['service']) ? $atts['service'] : 'SEO';
        $location = !empty($atts['location']) ? $atts['location'] : 'your area';
        $phone = !empty($atts['phone']) ? $atts['phone'] : get_option('seo_lead_pages_phone_number', '+1234567890');
        $email = !empty($atts['email']) ? $atts['email'] : get_option('seo_lead_pages_contact_email', 'info@example.com');
        $website = !empty($atts['website']) ? $atts['website'] : get_option('seo_lead_pages_website', 'https://example.com');
        
        ob_start();
        ?>
        <style>
        .seo-lead-form {
            background: <?php echo esc_attr($form_background_color); ?> !important;
            border: 1px solid <?php echo esc_attr($form_border_color); ?> !important;
            color: <?php echo esc_attr($form_text_color); ?> !important;
            border-radius: <?php echo esc_attr($form_border_radius); ?>px !important;
            padding: <?php echo esc_attr($form_padding); ?>px !important;
            width: <?php echo esc_attr($form_width); ?> !important;
            text-align: <?php echo esc_attr($form_alignment); ?> !important;
        }
        
        .seo-lead-form h3 {
            color: <?php echo esc_attr($form_title_color); ?> !important;
        }
        
        .seo-lead-form .form-row input,
        .seo-lead-form .form-row textarea {
            background: <?php echo esc_attr($input_background_color); ?> !important;
            border: 2px solid <?php echo esc_attr($input_border_color); ?> !important;
            color: <?php echo esc_attr($input_text_color); ?> !important;
        }
        
        .seo-lead-form .submit-button {
            background: <?php echo esc_attr($button_background_color); ?> !important;
            color: <?php echo esc_attr($button_text_color); ?> !important;
        }
        
        .seo-lead-form .submit-button:hover {
            background: <?php echo esc_attr($button_hover_color); ?> !important;
        }
        </style>
        
        <div class="seo-lead-form">
            <h3>🎯 <?php echo esc_html($form_title); ?> <?php echo esc_html($location); ?></h3>
            
            <?php echo $contact_info; ?>
            
            <form id="lead-form-<?php echo uniqid(); ?>" class="lead-form">
                <div class="form-row">
                    <label for="name"><?php echo esc_html($name_label); ?> *</label>
                    <input type="text" id="name" name="name" required>
                </div>
                
                <div class="form-row">
                    <label for="email"><?php echo esc_html($email_label); ?> *</label>
                    <input type="email" id="email" name="email" required>
                </div>
                
                <div class="form-row">
                    <label for="phone"><?php echo esc_html($phone_label); ?></label>
                    <input type="tel" id="phone" name="phone">
                </div>
                
                <div class="form-row">
                    <label for="company"><?php echo esc_html($company_label); ?></label>
                    <input type="text" id="company" name="company">
                </div>
                
                <div class="form-row">
                    <label for="message"><?php echo esc_html($message_label); ?></label>
                    <textarea id="message" name="message" placeholder="<?php echo esc_attr($message_placeholder); ?>"></textarea>
                </div>
                
                <button type="submit" class="submit-button">🚀 <?php echo esc_html($submit_button); ?></button>
                
                <input type="hidden" name="service" value="<?php echo esc_attr($service); ?>">
                <input type="hidden" name="location" value="<?php echo esc_attr($location); ?>">
                <input type="hidden" name="action" value="submit_lokaal_lead_fixed">
                <input type="hidden" name="nonce" value="<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>">
                
                <div class="form-message" style="display: none;"></div>
            </form>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('.lead-form').on('submit', function(e) {
                e.preventDefault();
                
                var $form = $(this);
                var $message = $form.find('.form-message');
                var $submitButton = $form.find('.submit-button');
                
                $submitButton.prop('disabled', true).text('<?php echo esc_js(__('Sending...', 'geo-local-seo-generator')); ?>');
                $message.hide();
                
                $.post(seo_lead_pages_ajax.ajax_url, $form.serialize(), function(response) {
                    if (response.success) {
                        $message.removeClass('error').addClass('success').html('<?php echo esc_js(__('Thank you! We will contact you as soon as possible.', 'geo-local-seo-generator')); ?>').show();
                        $form[0].reset();
                    } else {
                        $message.removeClass('success').addClass('error').html('<?php echo esc_js(__('An error occurred. Please try again.', 'geo-local-seo-generator')); ?>').show();
                    }
                }).fail(function() {
                    $message.removeClass('success').addClass('error').html('<?php echo esc_js(__('An error occurred. Please try again.', 'geo-local-seo-generator')); ?>').show();
                }).always(function() {
                    $submitButton.prop('disabled', false).text('🚀 <?php echo esc_js($submit_button); ?>');
                });
            });
        });
        </script>
        <?php
        return ob_get_clean();
    }
}
?>
