<?php
/**
 * Database Class for SEO Lead Pages Generator
 * Handles database operations and table creation
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class SEO_Lead_Pages_Database {
    
    public function __construct() {
        add_action('init', array($this, 'check_tables'));
    }
    
    public function check_tables() {
        if (get_option('seo_lead_pages_db_version') != '1.1') {
            $this->create_tables();
            update_option('seo_lead_pages_db_version', '1.1');
        }
    }
    
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Services table
        $table_services = $wpdb->prefix . 'seo_lead_pages_services';
        $sql_services = "CREATE TABLE $table_services (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            slug varchar(255) NOT NULL,
            description text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY slug (slug)
        ) $charset_collate;";
        
        // Locations table
        $table_locations = $wpdb->prefix . 'seo_lead_pages_locations';
        $sql_locations = "CREATE TABLE $table_locations (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            province varchar(255) NOT NULL,
            name varchar(255) NOT NULL,
            population int(11) DEFAULT 0,
            landmarks text,
            country varchar(255) DEFAULT 'Netherlands',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY province (province),
            KEY name (name)
        ) $charset_collate;";
        
        // Leads table
        $table_leads = $wpdb->prefix . 'seo_lead_pages_leads';
        $sql_leads = "CREATE TABLE $table_leads (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            email varchar(255) NOT NULL,
            phone varchar(50),
            company varchar(255),
            service varchar(255),
            location varchar(255),
            message text,
            status varchar(50) DEFAULT 'new',
            page_id bigint(20),
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY status (status),
            KEY created_at (created_at),
            KEY page_id (page_id)
        ) $charset_collate;";
        
        // Page views table
        $table_page_views = $wpdb->prefix . 'seo_lead_pages_page_views';
        $sql_page_views = "CREATE TABLE $table_page_views (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            page_id bigint(20) NOT NULL,
            page_slug varchar(255) NOT NULL,
            province varchar(255),
            views int(11) DEFAULT 1,
            view_date date NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY page_id (page_id),
            KEY page_slug (page_slug),
            KEY province (province),
            KEY view_date (view_date)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        dbDelta($sql_services);
        dbDelta($sql_locations);
        dbDelta($sql_leads);
        dbDelta($sql_page_views);
        
        // Insert some default services if table is empty
        if ($wpdb->get_var("SELECT COUNT(*) FROM $table_services") == 0) {
            $default_services = array(
                array('name' => 'SEO Optimalisatie', 'slug' => 'seo-optimalisatie'),
                array('name' => 'Webdesign', 'slug' => 'webdesign'),
                array('name' => 'Content Marketing', 'slug' => 'content-marketing'),
                array('name' => 'Google Ads', 'slug' => 'google-ads'),
                array('name' => 'Social Media Marketing', 'slug' => 'social-media-marketing')
            );
            
            foreach ($default_services as $service) {
                $wpdb->insert($table_services, $service);
            }
        }
        
        // Insert some default locations if table is empty
        if ($wpdb->get_var("SELECT COUNT(*) FROM $table_locations") == 0) {
            $default_locations = array(
                array('province' => 'Noord-Holland', 'name' => 'Amsterdam', 'population' => 872680),
                array('province' => 'Zuid-Holland', 'name' => 'Rotterdam', 'population' => 651446),
                array('province' => 'Zuid-Holland', 'name' => 'Den Haag', 'population' => 548320),
                array('province' => 'Noord-Brabant', 'name' => 'Eindhoven', 'population' => 235691),
                array('province' => 'Gelderland', 'name' => 'Arnhem', 'population' => 164096)
            );
            
            foreach ($default_locations as $location) {
                $wpdb->insert($table_locations, $location);
            }
        }
    }
    
    public function get_services() {
        global $wpdb;
        $table = $wpdb->prefix . 'seo_lead_pages_services';
        return $wpdb->get_results("SELECT * FROM $table ORDER BY name ASC");
    }
    
    public function get_locations($province = null) {
        global $wpdb;
        $table = $wpdb->prefix . 'seo_lead_pages_locations';
        
        if ($province) {
            return $wpdb->get_results($wpdb->prepare(
                "SELECT * FROM $table WHERE province = %s ORDER BY name ASC",
                $province
            ));
        }
        
        return $wpdb->get_results("SELECT * FROM $table ORDER BY province ASC, name ASC");
    }
    
    public function get_provinces() {
        global $wpdb;
        $table = $wpdb->prefix . 'seo_lead_pages_locations';
        return $wpdb->get_col("SELECT DISTINCT province FROM $table ORDER BY province ASC");
    }
    
    public function save_lead($data) {
        global $wpdb;
        $table = $wpdb->prefix . 'seo_lead_pages_leads';
        
        $result = $wpdb->insert($table, array(
            'name' => sanitize_text_field($data['name']),
            'email' => sanitize_email($data['email']),
            'phone' => sanitize_text_field($data['phone']),
            'company' => sanitize_text_field($data['company'] ?? ''),
            'service' => sanitize_text_field($data['service']),
            'location' => sanitize_text_field($data['location']),
            'message' => sanitize_textarea_field($data['message']),
            'page_id' => isset($data['page_id']) ? intval($data['page_id']) : null,
            'status' => 'new'
        ));
        
        if ($result) {
            // Send email notification
            $this->send_lead_notification($data);
            return $wpdb->insert_id;
        }
        
        return false;
    }
    
    private function send_lead_notification($data) {
        $to = get_option('seo_lead_pages_contact_email', get_option('admin_email'));
        $subject = 'Nieuwe lead ontvangen - ' . get_bloginfo('name');
        
        $message = "Er is een nieuwe lead ontvangen:\n\n";
        $message .= "Naam: " . $data['name'] . "\n";
        $message .= "Email: " . $data['email'] . "\n";
        $message .= "Telefoon: " . ($data['phone'] ?? '') . "\n";
        $message .= "Bedrijf: " . ($data['company'] ?? '') . "\n";
        $message .= "Dienst: " . $data['service'] . "\n";
        $message .= "Locatie: " . $data['location'] . "\n";
        $message .= "Bericht: " . $data['message'] . "\n\n";
        $message .= "Deze lead is automatisch opgeslagen in het WordPress admin panel.";
        
        wp_mail($to, $subject, $message);
    }
    
    public function record_page_view($page_id, $page_slug, $province = null) {
        global $wpdb;
        $table = $wpdb->prefix . 'seo_lead_pages_page_views';
        
        $today = date('Y-m-d');
        
        // Check if view already exists for today
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE page_id = %d AND view_date = %s",
            $page_id, $today
        ));
        
        if ($existing) {
            // Update existing view count
            $wpdb->update(
                $table,
                array('views' => $wpdb->get_var($wpdb->prepare(
                    "SELECT views + 1 FROM $table WHERE id = %d",
                    $existing
                ))),
                array('id' => $existing),
                array('%d'),
                array('%d')
            );
        } else {
            // Insert new view record
            $wpdb->insert($table, array(
                'page_id' => $page_id,
                'page_slug' => $page_slug,
                'province' => $province,
                'view_date' => $today
            ));
        }
    }
}
?>
