<?php
/**
 * Admin functionality for GEO Local SEO Lead Pages Generator
 * Free version - Premium features moved to separate addon
 */

if (!defined('ABSPATH')) {
    exit;
}

class GEO_Local_SEO_Admin {
    
    public $premium_active = false;
    
    public function __construct() {
        // Initialize premium_active property
        $this->premium_active = false;
        
        // Check if premium addon is active
        $this->check_premium_status();
        
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // AJAX handlers for free version
        add_action('wp_ajax_generate_lead_page', array($this, 'ajax_generate_lead_page'));
        add_action('wp_ajax_cleanup_test_data', array($this, 'ajax_cleanup_test_data'));
        add_action('wp_ajax_export_leads_csv', array($this, 'ajax_export_leads_csv'));
        add_action('wp_ajax_generate_sitemap', array($this, 'ajax_generate_sitemap'));
        add_action('wp_ajax_check_seo_issues', array($this, 'ajax_check_seo_issues'));
        add_action('wp_ajax_bulk_seo_optimization', array($this, 'ajax_bulk_seo_optimization'));
        add_action('wp_ajax_bulk_generate_pages', array($this, 'ajax_bulk_generate_pages'));
        add_action('wp_ajax_import_locations_csv', array($this, 'ajax_import_locations_csv'));
        add_action('wp_ajax_save_template', array($this, 'ajax_save_template'));
        add_action('wp_ajax_delete_template', array($this, 'ajax_delete_template'));
        add_action('wp_ajax_get_template_library', array($this, 'ajax_get_template_library'));
        
        // Lead management AJAX handlers
        add_action('wp_ajax_update_lead_status', array($this, 'ajax_update_lead_status'));
        add_action('wp_ajax_delete_lead', array($this, 'ajax_delete_lead'));
        add_action('wp_ajax_bulk_delete_leads', array($this, 'ajax_bulk_delete_leads'));
        
        // Location management AJAX handlers
        add_action('wp_ajax_add_location_manually', array($this, 'ajax_add_location_manually'));
        add_action('wp_ajax_add_location_and_generate_page', array($this, 'ajax_add_location_and_generate_page'));
        add_action('wp_ajax_get_saved_locations', array($this, 'ajax_get_saved_locations'));
        
        // Page generator AJAX handlers
        add_action('wp_ajax_quick_generate_pages', array($this, 'ajax_quick_generate_pages'));
        add_action('wp_ajax_get_saved_locations_for_generator', array($this, 'ajax_get_saved_locations_for_generator'));
        add_action('wp_ajax_get_saved_templates_for_generator', array($this, 'ajax_get_saved_templates_for_generator'));
        add_action('wp_ajax_get_templates_overview', array($this, 'ajax_get_templates_overview'));
        
        // Dashboard AJAX handlers
        add_action('wp_ajax_get_dashboard_stats', array($this, 'ajax_get_dashboard_stats'));
        
        // Additional location management AJAX handlers
        add_action('wp_ajax_generate_page_from_saved_location', array($this, 'ajax_generate_page_from_saved_location'));
        add_action('wp_ajax_delete_saved_location', array($this, 'ajax_delete_saved_location'));
        add_action('wp_ajax_delete_imported_location', array($this, 'ajax_delete_imported_location'));
        
        // Bulk delete AJAX handlers
        add_action('wp_ajax_get_bulk_delete_countries', array($this, 'ajax_get_bulk_delete_countries'));
        add_action('wp_ajax_get_bulk_delete_provinces', array($this, 'ajax_get_bulk_delete_provinces'));
        add_action('wp_ajax_get_bulk_delete_count', array($this, 'ajax_get_bulk_delete_count'));
        add_action('wp_ajax_bulk_delete_locations', array($this, 'ajax_bulk_delete_locations'));
        
        // Saved locations bulk delete AJAX handlers
        add_action('wp_ajax_get_saved_locations_for_bulk_delete', array($this, 'ajax_get_saved_locations_for_bulk_delete'));
        add_action('wp_ajax_bulk_delete_saved_locations', array($this, 'ajax_bulk_delete_saved_locations'));
        
        // Bulk operations AJAX handlers are handled by premium addon
        
        // Lead form AJAX handlers
        add_action('wp_ajax_submit_lokaal_lead_fixed', array($this, 'ajax_submit_lead'));
        add_action('wp_ajax_nopriv_submit_lokaal_lead_fixed', array($this, 'ajax_submit_lead'));
        
        // Lead Management AJAX handlers
        add_action('wp_ajax_get_lead_stats', array($this, 'ajax_get_lead_stats'));
        add_action('wp_ajax_get_leads_table', array($this, 'ajax_get_leads_table'));
        
        // Add hooks for premium addon integration
        add_action('geo_local_seo_premium_loaded', array($this, 'on_premium_loaded'));
        add_filter('geo_local_seo_available_templates', array($this, 'get_available_templates'));
        add_filter('geo_local_seo_is_premium_active', array($this, 'is_premium_active'));
    }
    
    private function check_premium_status() {
        // Always initialize and reset to false first
        if (!isset($this->premium_active)) {
            $this->premium_active = false;
        }
        $this->premium_active = false;
        
        error_log('GEO Local SEO: Checking premium status...');
        
        // First check if premium addon is installed and active
        $premium_addon_active = false;
        
        // Check if premium addon class exists
        if (class_exists('GEO_Local_SEO_Premium')) {
            error_log('GEO Local SEO: Premium addon class found');
            $premium_addon_active = true;
        }
        
        // Check if premium bulk operations class exists
        if (class_exists('GEO_Local_SEO_Premium_Bulk_Operations')) {
            error_log('GEO Local SEO: Premium bulk operations class found');
            $premium_addon_active = true;
        }
        
        // Also check if the premium addon plugin is active
        if (is_plugin_active('GEO-Local-SEO-Lead-Pages-Generator-Premium/geo-local-seo-premium.php')) {
            error_log('GEO Local SEO: Premium plugin is active via is_plugin_active');
            $premium_addon_active = true;
        }
        
        // Check for premium addon in active plugins
        $active_plugins = get_option('active_plugins', array());
        foreach ($active_plugins as $plugin) {
            if (strpos($plugin, 'GEO-Local-SEO-Lead-Pages-Generator-Premium') !== false) {
                error_log('GEO Local SEO: Premium plugin found in active plugins: ' . $plugin);
                $premium_addon_active = true;
                break;
            }
        }
        
        // Check if premium addon is loaded via hook
        if (did_action('geo_local_seo_premium_loaded')) {
            error_log('GEO Local SEO: Premium addon loaded via hook');
            $premium_addon_active = true;
        }
        
        // If premium addon is not active, premium features are not available
        if (!$premium_addon_active) {
            error_log('GEO Local SEO: Premium addon not detected, premium_active = false');
            return;
        }
        
        // Now check if license is valid
        if (class_exists('GEO_Local_SEO_Premium_License_Manager')) {
            $license_manager = new GEO_Local_SEO_Premium_License_Manager();
            $license_status = $license_manager->get_license_status();
            
            if ($license_status['is_valid']) {
                error_log('GEO Local SEO: Premium license is valid, premium_active = true');
                $this->premium_active = true;
            } else {
                error_log('GEO Local SEO: Premium addon active but license invalid, premium_active = false');
                error_log('GEO Local SEO: License status: ' . print_r($license_status, true));
            }
        } else {
            error_log('GEO Local SEO: Premium addon active but license manager not found, premium_active = false');
        }
    }
    
    public function add_admin_menu() {
        // Ensure premium status is checked when menu is loaded
        $this->check_premium_status();
        
        add_menu_page(
            'GEO - Local SEO',
            'GEO - Local SEO',
            'manage_options',
            'seo-lead-pages',
            array($this, 'dashboard_page'),
            'dashicons-location-alt',
            30
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'Getting Started',
            'Getting Started',
            'manage_options',
            'seo-lead-pages',
            array($this, 'dashboard_page')
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'Locations Manager',
            'Locations Manager',
            'manage_options',
            'seo-lead-pages-locations',
            array($this, 'locations_manager_page')
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'Template Maker',
            'Template Maker',
            'manage_options',
            'seo-lead-pages-templates',
            array($this, 'template_maker_page')
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'Form Settings',
            'Form Settings',
            'manage_options',
            'seo-lead-pages-form-settings',
            array($this, 'form_settings_page')
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'Shortcode Settings',
            'Shortcode Settings',
            'manage_options',
            'seo-lead-pages-shortcodes',
            array($this, 'shortcode_settings_page')
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'Page Generator',
            'Page Generator',
            'manage_options',
            'seo-lead-pages-generator',
            array($this, 'admin_generator')
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'Lead Management',
            'Lead Management',
            'manage_options',
            'seo-lead-pages-leads',
            array($this, 'lead_management_page')
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'Analytics',
            'Analytics',
            'manage_options',
            'seo-lead-pages-analytics',
            array($this, 'analytics_page')
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'Premium Addon',
            'Premium Addon',
            'manage_options',
            'seo-lead-pages-premium-info',
            array($this, 'premium_info_page')
        );
        
        add_submenu_page(
            'seo-lead-pages',
            'About This Plugin',
            'About This Plugin',
            'manage_options',
            'seo-lead-pages-about',
            array($this, 'about_plugin_page')
        );
    }
    
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'seo-lead-pages') === false) {
            return;
        }
        
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-sortable');
        wp_enqueue_style('wp-admin');
        wp_enqueue_media();
        
        wp_enqueue_style(
            'seo-lead-pages-admin',
            plugin_dir_url(__FILE__) . '../css/admin.css',
            array(),
            '1.0.0'
        );
        
        wp_enqueue_script(
            'seo-lead-pages-admin',
            plugin_dir_url(__FILE__) . '../js/admin.js',
            array('jquery'),
            '1.0.0',
            true
        );
        
        wp_localize_script('seo-lead-pages-admin', 'seoLeadPages', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seo_lead_pages_nonce')
        ));
        
        // Also define global ajaxurl for compatibility
        wp_localize_script('seo-lead-pages-admin', 'ajaxurl', admin_url('admin-ajax.php'));
    }
    
    // Premium Addon Notice
    public function premium_addon_notice() {
        ?>
        <div class="notice notice-info" style="border-left-color: #0073aa;">
            <p><strong>🚀 Want more features?</strong> Check out our <a href="<?php echo admin_url('admin.php?page=seo-lead-pages-premium-info'); ?>">Premium Addon</a> for advanced analytics, bulk operations, AI content management, and more!</p>
        </div>
        <?php
    }
    
    // Admin Pages
    public function dashboard_page() {
        // Ensure premium status is checked when page is loaded
        $this->check_premium_status();
        
        $this->premium_addon_notice();
        ?>
        <div class="wrap">
            <h1>🚀 Welcome to GEO Local SEO Lead Pages Generator</h1>
            <p>Generate unlimited SEO-optimized landing pages for your local business locations.</p>
            
            <!-- Quick Stats -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin: 20px 0;">
                <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; text-align: center;">
                    <h3 style="margin: 0; color: #0073aa;">📍 Saved Locations</h3>
                    <p style="font-size: 24px; margin: 10px 0; font-weight: bold;" id="locations-count">Loading...</p>
                </div>
                <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; text-align: center;">
                    <h3 style="margin: 0; color: #0073aa;">📄 Generated Pages</h3>
                    <p style="font-size: 24px; margin: 10px 0; font-weight: bold;" id="pages-count">Loading...</p>
                </div>
                <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; text-align: center;">
                    <h3 style="margin: 0; color: #0073aa;">📊 Total Leads</h3>
                    <p style="font-size: 24px; margin: 10px 0; font-weight: bold;" id="leads-count">Loading...</p>
                </div>
                <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; text-align: center;">
                    <h3 style="margin: 0; color: #0073aa;">🎨 Templates</h3>
                    <p style="font-size: 24px; margin: 10px 0; font-weight: bold;" id="templates-count">Loading...</p>
                </div>
            </div>
            
            <!-- Getting Started Guide -->
            <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h2 style="margin-top: 0; color: #0073aa;">📚 How to Use This Plugin</h2>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px;">
                    <div>
                        <h3>🎯 Step 1: Add Your Locations</h3>
                        <p>Start by adding your business locations using the <strong>Locations Manager</strong>:</p>
                        <ul>
                            <li>Click on <strong>"Locations Manager"</strong> in the menu</li>
                            <li>Fill in city, province/state, and country</li>
                            <li>Add your business details (name, service, contact info)</li>
                            <li>Save locations for reuse</li>
                        </ul>
                        <a href="?page=seo-lead-pages-locations" class="button button-primary">Go to Locations Manager</a>
                    </div>
                    
                    <div>
                        <h3>🎨 Step 2: Create Templates (Optional)</h3>
                        <p>Customize your page content using the <strong>Template Maker</strong>:</p>
                        <ul>
                            <li>Click on <strong>"Template Maker"</strong> in the menu</li>
                            <li>Use spintax for content variation: <code>{word1|word2|word3}</code></li>
                            <li>Use shortcodes: <code>[service_name]</code>, <code>[location_name]</code></li>
                            <li>Save templates for reuse</li>
                        </ul>
                        <a href="?page=seo-lead-pages-templates" class="button button-primary">Go to Template Maker</a>
                    </div>
                    
                    <div>
                        <h3>📝 Step 3: Customize Your Forms</h3>
                        <p>Customize the appearance and text of your lead forms to match your brand and language preferences:</p>
                        <ul>
                            <li>Click on <strong>"Form Settings"</strong> in the menu</li>
                            <li>Change form field labels (Name, Email, Phone, etc.)</li>
                            <li>Customize form title and submit button text</li>
                            <li>Choose which contact info to display (phone, email, website)</li>
                            <li>Set custom message placeholder text</li>
                        </ul>
                        <p><strong>💡 Pro Tip:</strong> You can hide contact information to focus visitors on filling out the form instead of calling directly.</p>
                        <a href="?page=seo-lead-pages-form-settings" class="button button-primary">Configure Form Settings</a>
                    </div>
                    
                    <div>
                        <h3>⚙️ Step 4: Configure Shortcode Settings</h3>
                        <p>Set up your default content using the <strong>Shortcode Settings</strong>:</p>
                        <ul>
                            <li>Click on <strong>"Shortcode Settings"</strong> in the menu</li>
                            <li>Configure your service name, business name, and contact details</li>
                            <li>Set default values for phone, email, and website</li>
                            <li><strong>Required:</strong> These settings must be configured before generating pages</li>
                        </ul>
                        <a href="?page=seo-lead-pages-shortcodes" class="button button-primary">Configure Shortcode Settings</a>
                    </div>
                    
                    <div>
                        <h3>🚀 Step 5: Generate Pages</h3>
                        <p>Create SEO-optimized landing pages using the <strong>Page Generator</strong>:</p>
                        <ul>
                            <li>Click on <strong>"Page Generator"</strong> in the menu</li>
                            <li>Select from your saved locations</li>
                            <li>Choose a template (or use default)</li>
                            <li><strong>New:</strong> Customize URL structure with custom slug templates</li>
                            <li>Generate single pages or bulk generate</li>
                        </ul>
                        <a href="?page=seo-lead-pages-generator" class="button button-primary">Go to Page Generator</a>
                    </div>
                    
                    <div>
                        <h3>📊 Step 6: Track Performance</h3>
                        <p>Monitor your results using the <strong>Analytics</strong> dashboard:</p>
                        <ul>
                            <li>Click on <strong>"Analytics"</strong> in the menu</li>
                            <li>View lead statistics and page performance</li>
                            <li>Manage collected leads</li>
                            <li>Export data for analysis</li>
                        </ul>
                        <a href="?page=seo-lead-pages-analytics" class="button button-primary">Go to Analytics</a>
                    </div>
                    
                    <div>
                        <h3>📋 Step 7: Manage Your Leads</h3>
                        <p>Track and manage leads from your landing pages using the <strong>Lead Management</strong> dashboard:</p>
                        <ul>
                            <li>Click on <strong>"Lead Management"</strong> in the menu</li>
                            <li>View all collected leads in one place</li>
                            <li>Update lead status (Open, In Progress, Rejected, Completed)</li>
                            <li>Delete individual leads or bulk delete</li>
                            <li>Export leads to CSV for external analysis</li>
                        </ul>
                        <a href="?page=seo-lead-pages-leads" class="button button-primary">Go to Lead Management</a>
                    </div>
                </div>
            </div>
            
            <!-- New Features -->
            <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h2 style="margin-top: 0; color: #0073aa;">🆕 New Features</h2>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px;">
                    <div>
                        <h3>🔗 Custom URL Slug Templates</h3>
                        <p><strong>Full control over your page URLs!</strong> Create custom URL structures in the Page Generator:</p>
                        <ul>
                            <li><strong>Predefined Templates:</strong> Choose from 5 built-in URL structures</li>
                            <li><strong>Custom Templates:</strong> Create your own URL structure using variables</li>
                            <li><strong>Live Preview:</strong> See exactly how your URLs will look</li>
                            <li><strong>SEO Optimized:</strong> Perfect for local SEO strategies</li>
                        </ul>
                        
                        <h4>Available Variables:</h4>
                        <ul style="font-family: monospace; background: #f8f9fa; padding: 10px; border-radius: 4px;">
                            <li><code>[service_name]</code> - Your service name</li>
                            <li><code>[location_name]</code> - City name</li>
                            <li><code>[province_name]</code> - Province/state</li>
                            <li><code>[business_name]</code> - Your business name</li>
                            <li><code>[country]</code> - Country name</li>
                        </ul>
                        
                        <h4>Example Custom Templates:</h4>
                        <ul style="font-family: monospace; background: #f8f9fa; padding: 10px; border-radius: 4px;">
                            <li><code>[service_name]-[location_name]</code> → "seo-diensten-amsterdam"</li>
                            <li><code>[location_name]-[service_name]</code> → "amsterdam-seo-diensten"</li>
                            <li><code>professionele-[service_name]-in-[location_name]</code> → "professionele-seo-diensten-in-amsterdam"</li>
                        </ul>
                    </div>
                    
                    <div>
                        <h3>⚙️ Shortcode Settings Integration</h3>
                        <p><strong>Centralized content management!</strong> Configure default values for all your pages:</p>
                        <ul>
                            <li><strong>Service Name:</strong> Set your default service name</li>
                            <li><strong>Business Name:</strong> Your company name</li>
                            <li><strong>Contact Details:</strong> Phone, email, website defaults</li>
                            <li><strong>Required Settings:</strong> Must be configured before generating pages</li>
                        </ul>
                        
                        <h4>How It Works:</h4>
                        <ol>
                            <li>Configure defaults in <strong>Shortcode Settings</strong></li>
                            <li>Use location-specific data when available</li>
                            <li>Fall back to your configured defaults</li>
                            <li>Generate consistent, branded pages</li>
                        </ol>
                        
                        <div style="background: #e7f3ff; border: 1px solid #b8daff; padding: 15px; border-radius: 5px; margin-top: 15px;">
                            <h4 style="margin-top: 0;">💡 Pro Tips:</h4>
                            <ul style="margin-bottom: 0;">
                                <li>Use <strong>Page Generator</strong> for quick URL overrides</li>
                                <li>Set <strong>Shortcode Settings</strong> for permanent defaults</li>
                                <li>Custom templates work in both free and premium versions</li>
                                <li>Live preview helps you choose the best URL structure</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Quick Actions -->
            <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h2 style="margin-top: 0; color: #0073aa;">⚡ Quick Actions</h2>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px;">
                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px;">
                        <h4 style="margin-top: 0;">📍 Add Your First Location</h4>
                        <p>Start by adding your business locations to generate location-specific pages.</p>
                        <a href="?page=seo-lead-pages-locations" class="button">Add Location</a>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px;">
                        <h4 style="margin-top: 0;">📝 Customize Forms</h4>
                        <p>Personalize your lead forms with custom text and settings.</p>
                        <a href="?page=seo-lead-pages-form-settings" class="button">Form Settings</a>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px;">
                        <h4 style="margin-top: 0;">⚙️ Configure Shortcodes</h4>
                        <p>Set up your default service name, business details, and contact info.</p>
                        <a href="?page=seo-lead-pages-shortcodes" class="button">Shortcode Settings</a>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px;">
                        <h4 style="margin-top: 0;">🚀 Generate Pages</h4>
                        <p>Create SEO-optimized landing pages with custom URL structures.</p>
                        <a href="?page=seo-lead-pages-generator" class="button">Generate Pages</a>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px;">
                        <h4 style="margin-top: 0;">📋 Manage Leads</h4>
                        <p>Track and manage leads from your landing pages.</p>
                        <a href="?page=seo-lead-pages-leads" class="button">Lead Management</a>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px;">
                        <h4 style="margin-top: 0;">📊 View Analytics</h4>
                        <p>Check your lead statistics and page performance.</p>
                        <a href="?page=seo-lead-pages-analytics" class="button">View Analytics</a>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 20px; border-radius: 8px;">
                        <h4 style="margin-top: 0;">🎨 Create Templates</h4>
                        <p>Design custom page templates with spintax and shortcodes.</p>
                        <a href="?page=seo-lead-pages-templates" class="button">Create Template</a>
                    </div>
                </div>
            </div>
            
            <!-- Tips & Tricks -->
            <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h2 style="margin-top: 0; color: #0073aa;">💡 Tips & Tricks</h2>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 20px;">
                    <div>
                        <h4>🎯 SEO Best Practices</h4>
                        <ul>
                            <li>Use specific location names (city + province/state)</li>
                            <li>Include your main service in page titles</li>
                            <li>Add contact information for local SEO signals</li>
                            <li>Use spintax to create unique content variations</li>
                        </ul>
                    </div>
                    
                    <div>
                        <h4>🔧 Content Enhancement</h4>
                        <ul>
                            <li><strong>Spintax:</strong> <code>{Professional|Expert|Skilled}</code></li>
                            <li><strong>Shortcodes:</strong> <code>[service_name]</code>, <code>[location_name]</code></li>
                            <li><strong>Variations:</strong> <code>{plumbing|electrical|painting} services</code></li>
                            <li><strong>Time:</strong> <code>{24/7|around the clock|24 hours a day}</code></li>
                        </ul>
                    </div>
                    
                    <div>
                        <h4>📝 Form Optimization</h4>
                        <ul>
                            <li><strong>Hide Contact Info:</strong> Focus visitors on form completion</li>
                            <li><strong>Custom Labels:</strong> Match your brand voice and language</li>
                            <li><strong>Clear CTAs:</strong> Use action-oriented button text</li>
                            <li><strong>Required Fields:</strong> Only require essential information</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Load dashboard stats
            loadDashboardStats();
            
            function loadDashboardStats() {
                // Load locations count
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_dashboard_stats',
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#locations-count').text(response.data.locations);
                            $('#pages-count').text(response.data.pages);
                            $('#leads-count').text(response.data.leads);
                            $('#templates-count').text(response.data.templates);
                        }
                    }
                });
            }
        });
        </script>
        <?php
    }
    
    public function shortcode_settings_page() {
        // Handle form submission
        if (isset($_POST['submit']) && wp_verify_nonce($_POST['shortcode_settings_nonce'], 'shortcode_settings_action')) {
            // Save shortcode mappings
            $shortcode_mappings = array();
            $shortcodes = array('service_name', 'location_name', 'province_name', 'business_name', 'phone_number', 'contact_email', 'website_url');
            
            foreach ($shortcodes as $shortcode) {
                if (isset($_POST['shortcode_' . $shortcode])) {
                    $shortcode_mappings[$shortcode] = sanitize_text_field($_POST['shortcode_' . $shortcode]);
                }
            }
            
            update_option('seo_lead_pages_shortcode_mappings', $shortcode_mappings);
            
            // Debug: Log what was saved
            error_log('GEO Local SEO: Shortcode settings saved: ' . print_r($shortcode_mappings, true));
            
            echo '<div class="notice notice-success"><p>Shortcode settings saved successfully!</p></div>';
        }
        
        // Get current settings
        $shortcode_mappings = get_option('seo_lead_pages_shortcode_mappings', array());
        
        // Debug: Log current shortcode mappings
        error_log('GEO Local SEO: Current shortcode mappings on settings page: ' . print_r($shortcode_mappings, true));
        
        // Debug: Show what will be used for slug generation
        if (!empty($shortcode_mappings['service_name'])) {
            error_log('GEO Local SEO: Slug will use service_name: "' . $shortcode_mappings['service_name'] . '"');
        }
        if (!empty($shortcode_mappings['slug_template'])) {
            error_log('GEO Local SEO: Slug will use template: "' . $shortcode_mappings['slug_template'] . '"');
        }
        
        // Default values
        $defaults = array(
            'service_name' => 'SEO Diensten',
            'location_name' => 'Amsterdam',
            'province_name' => 'Noord-Holland',
            'business_name' => get_option('blogname', 'Uw Bedrijf'),
            'phone_number' => '+31 20 123 4567',
            'contact_email' => get_option('admin_email'),
            'website_url' => get_option('home'),
            'slug_template' => 'service_in_location'
        );
        
        ?>
        <div class="wrap">
            <h1>Shortcode Settings</h1>
            <p>Configure what each shortcode should be replaced with in your templates.</p>
            
            <form method="post" action="">
                <?php wp_nonce_field('shortcode_settings_action', 'shortcode_settings_nonce'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">[service_name] <span style="color: red;">*</span></th>
                        <td>
                            <input type="text" name="shortcode_service_name" value="<?php echo esc_attr(isset($shortcode_mappings['service_name']) ? $shortcode_mappings['service_name'] : $defaults['service_name']); ?>" class="regular-text" required />
                            <p class="description">Default service name to use when [service_name] is not specified in location data <strong>(Required)</strong></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">[location_name]</th>
                        <td>
                            <input type="text" name="shortcode_location_name" value="<?php echo esc_attr(isset($shortcode_mappings['location_name']) ? $shortcode_mappings['location_name'] : $defaults['location_name']); ?>" class="regular-text" />
                            <p class="description">Default city name to use when [location_name] is not specified</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">[province_name]</th>
                        <td>
                            <input type="text" name="shortcode_province_name" value="<?php echo esc_attr(isset($shortcode_mappings['province_name']) ? $shortcode_mappings['province_name'] : $defaults['province_name']); ?>" class="regular-text" />
                            <p class="description">Default province/state name to use when [province_name] is not specified</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">[business_name] <span style="color: red;">*</span></th>
                        <td>
                            <input type="text" name="shortcode_business_name" value="<?php echo esc_attr(isset($shortcode_mappings['business_name']) ? $shortcode_mappings['business_name'] : $defaults['business_name']); ?>" class="regular-text" required />
                            <p class="description">Default business name to use when [business_name] is not specified <strong>(Required)</strong></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">[phone_number] <span style="color: red;">*</span></th>
                        <td>
                            <input type="text" name="shortcode_phone_number" value="<?php echo esc_attr(isset($shortcode_mappings['phone_number']) ? $shortcode_mappings['phone_number'] : $defaults['phone_number']); ?>" class="regular-text" required />
                            <p class="description">Default phone number to use when [phone_number] is not specified <strong>(Required)</strong></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">[contact_email] <span style="color: red;">*</span></th>
                        <td>
                            <input type="email" name="shortcode_contact_email" value="<?php echo esc_attr(isset($shortcode_mappings['contact_email']) ? $shortcode_mappings['contact_email'] : $defaults['contact_email']); ?>" class="regular-text" required />
                            <p class="description">Default email address to use when [contact_email] is not specified <strong>(Required)</strong></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">[website_url]</th>
                        <td>
                            <input type="url" name="shortcode_website_url" value="<?php echo esc_attr(isset($shortcode_mappings['website_url']) ? $shortcode_mappings['website_url'] : $defaults['website_url']); ?>" class="regular-text" />
                            <p class="description">Default website URL to use when [website_url] is not specified</p>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <input type="submit" name="submit" class="button-primary" value="Save Shortcode Settings" />
                </p>
            </form>
            
            <?php if (!empty($shortcode_mappings)): ?>
            <div style="background: #e7f3ff; border: 1px solid #b8daff; padding: 15px; border-radius: 5px; margin-top: 20px;">
                <h4>🔍 Debug: Current Shortcode Settings</h4>
                <p>De volgende instellingen zijn opgeslagen in de database:</p>
                <pre style="background: #f8f9fa; padding: 10px; border-radius: 4px; overflow-x: auto;">
<?php echo esc_html(print_r($shortcode_mappings, true)); ?>
                </pre>
                <p><strong>Test:</strong> Probeer nu een pagina te genereren en check de debug logs om te zien of deze instellingen worden gebruikt.</p>
            </div>
            <?php endif; ?>
            
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin-top: 30px;">
                <h3>📋 Available Shortcodes</h3>
                <p>Use these shortcodes in your templates:</p>
                <ul>
                    <li><code>[service_name]</code> - Service name (e.g., "SEO Services")</li>
                    <li><code>[location_name]</code> - City name (e.g., "Amsterdam")</li>
                    <li><code>[province_name]</code> - Province/state name (e.g., "Noord-Holland")</li>
                    <li><code>[business_name]</code> - Your business name</li>
                    <li><code>[phone_number]</code> - Contact phone number</li>
                    <li><code>[contact_email]</code> - Contact email address</li>
                    <li><code>[website_url]</code> - Your website URL</li>
                    <li><code>[contact_form]</code> - Contact form placeholder</li>
                    <li><code>[seo_lead_form]</code> - Lead capture form</li>
                </ul>
                
                <h4>💡 How it works:</h4>
                <ol>
                    <li><strong>Location-specific data:</strong> If a location has specific data (e.g., business_name: "My Company"), that will be used</li>
                    <li><strong>Default fallback:</strong> If location data is empty, the values you set here will be used</li>
                    <li><strong>Template usage:</strong> Use shortcodes in your templates like: "Contact [business_name] for [service_name] in [location_name]"</li>
                    <li><strong>Premium compatibility:</strong> These settings work with both the free version and premium addon</li>
                </ol>
                
                <div style="background: #e7f3ff; border: 1px solid #b8daff; padding: 15px; border-radius: 5px; margin-top: 20px;">
                    <h4>🔧 Premium Addon Compatibility</h4>
                    <p>When the premium addon is active, these shortcode settings will be used as fallbacks for:</p>
                    <ul>
                        <li>✅ Bulk page generation</li>
                        <li>✅ Country/province imports</li>
                        <li>✅ Premium templates (plumber, electrician, etc.)</li>
                        <li>✅ Advanced SEO features</li>
                    </ul>
                    <p><strong>Priority order:</strong> Location data → Shortcode settings → Plugin defaults</p>
                </div>
            </div>
        </div>
        
        
        <?php
    }
    
    private function get_shortcode_description($shortcode) {
        $descriptions = array(
            'service_name' => 'Naam van je dienst (bijv. "SEO Services" of "Loodgieter Diensten")',
            'business_name' => 'Naam van je bedrijf',
            'phone_number' => 'Telefoonnummer voor contact',
            'contact_email' => 'Emailadres voor contact',
            'location_name' => 'Stad/plaats naam (optioneel)',
            'province_name' => 'Provincie naam (optioneel)',
            'website_url' => 'Website URL (optioneel)'
        );
        
        return isset($descriptions[$shortcode]) ? $descriptions[$shortcode] : 'Shortcode beschrijving';
    }
    
    public function admin_generator() {
        $this->premium_addon_notice();
        
        // Check if shortcode settings are configured
        $shortcode_mappings = get_option('seo_lead_pages_shortcode_mappings', array());
        $required_shortcodes = array('service_name', 'business_name', 'phone_number', 'contact_email');
        $missing_shortcodes = array();
        
        foreach ($required_shortcodes as $shortcode) {
            if (empty($shortcode_mappings[$shortcode])) {
                $missing_shortcodes[] = $shortcode;
            }
        }
        
        if (!empty($missing_shortcodes)) {
            echo '<div class="notice notice-warning"><p>';
            echo '<strong>⚠️ Shortcode instellingen vereist!</strong><br>';
            echo 'Voordat je pagina\'s kunt genereren, moet je eerst je shortcode instellingen configureren.<br>';
            echo '<strong>Ontbrekende instellingen:</strong> ' . implode(', ', $missing_shortcodes) . '<br>';
            echo '<a href="' . admin_url('admin.php?page=seo-lead-pages-shortcodes') . '" class="button button-primary">Configureer Shortcode Instellingen</a>';
            echo '</p></div>';
        }
        
        ?>
        <div class="wrap">
            <h1>Page Generator</h1>
            <p>Generate SEO-optimized landing pages using your saved locations and templates.</p>
            
            <?php if (!empty($missing_shortcodes)): ?>
            <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <h3>🚨 Shortcode Instellingen Vereist</h3>
                <p>Voordat je pagina's kunt genereren, moet je eerst de volgende shortcode instellingen configureren:</p>
                <ul>
                    <?php foreach ($missing_shortcodes as $shortcode): ?>
                    <li><code>[<?php echo esc_html($shortcode); ?>]</code> - <?php echo esc_html($this->get_shortcode_description($shortcode)); ?></li>
                    <?php endforeach; ?>
                </ul>
                <p><strong>Waarom is dit nodig?</strong> Deze instellingen bepalen hoe je bedrijfsgegevens worden weergegeven in gegenereerde pagina's.</p>
                <a href="<?php echo admin_url('admin.php?page=seo-lead-pages-shortcodes'); ?>" class="button button-primary">Configureer Nu</a>
            </div>
            <?php endif; ?>
            
            <!-- Quick Generation -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>🚀 Quick Page Generation</h3>
                <p>Select from your saved locations and templates to quickly generate pages.</p>
                
                
                <form id="quick-generate-form">
                    <?php wp_nonce_field('seo_lead_pages_nonce', 'nonce'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">Select Location *</th>
                            <td>
                                <select name="selected_location" id="location-select" class="regular-text" required>
                                    <option value="">Choose a saved location...</option>
                                </select>
                                <p class="description">No locations? <a href="?page=seo-lead-pages-locations">Add locations first</a></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Select Template</th>
                            <td>
                                <select name="template" id="template-select" class="regular-text">
                                    <option value="">Use default template</option>
                                </select>
                                <p class="description">No templates? <a href="?page=seo-lead-pages-templates">Create templates first</a></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">URL Structure</th>
                            <td>
                                <select name="url_logic" id="url-logic" class="regular-text">
                                    <option value="service_in_location">Service in Location (e.g., "SEO in Amsterdam")</option>
                                    <option value="service_location">Service Location (e.g., "SEO Amsterdam")</option>
                                    <option value="location_service">Location Service (e.g., "Amsterdam SEO")</option>
                                    <option value="service_in_location_province">Service in Location, Province (e.g., "SEO in Amsterdam, North Holland")</option>
                                    <option value="service_in_location_country">Service in Location, Country (e.g., "SEO in Amsterdam, Netherlands")</option>
                                    <option value="custom">🎨 Custom Template (volledig zelf instelbaar)</option>
                                </select>
                                <p class="description">Choose how the page URL and title should be structured</p>
                                <div id="slug-preview" style="background: #f0f0f1; padding: 10px; margin-top: 10px; border-radius: 4px;">
                                    <strong>Voorbeeld URL:</strong> <span id="slug-example">seo-diensten-in-amsterdam</span>
                                </div>
                            </td>
                        </tr>
                        <tr id="custom-slug-row" style="display: none;">
                            <th scope="row">Custom Slug Template</th>
                            <td>
                                <input type="text" name="custom_slug_template" id="custom-slug-template" class="regular-text" placeholder="[service_name]-in-[location_name]" />
                                <p class="description">
                                    <strong>Beschikbare variabelen:</strong><br>
                                    • <code>[service_name]</code> - Je dienst naam<br>
                                    • <code>[location_name]</code> - Locatie naam<br>
                                    • <code>[province_name]</code> - Provincie naam<br>
                                    • <code>[business_name]</code> - Bedrijfsnaam<br>
                                    • <code>[country]</code> - Land naam<br>
                                    <strong>Voorbeelden:</strong><br>
                                    • <code>[service_name]-[location_name]</code> → "seo-diensten-amsterdam"<br>
                                    • <code>[location_name]-[service_name]</code> → "amsterdam-seo-diensten"<br>
                                    • <code>[service_name]-diensten-[location_name]</code> → "seo-diensten-amsterdam"<br>
                                    • <code>professionele-[service_name]-in-[location_name]</code> → "professionele-seo-diensten-in-amsterdam"
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Generate Multiple Pages</th>
                            <td>
                                <label>
                                    <input type="checkbox" name="generate_multiple" id="generate-multiple" />
                                    Generate pages for all saved locations
                                </label>
                                <p class="description">This will create pages for all your saved locations using the selected template.</p>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <input type="submit" class="button-primary" value="Generate Page(s)" />
                        <span id="generation-status"></span>
                    </p>
                </form>
                
                <div id="quick-generation-result"></div>
            </div>
            
            <?php if ($this->get_premium_active()): ?>
            <!-- Bulk Operations (Premium) -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>⚡ Bulk Operations</h3>
                <p>Generate multiple pages at once with advanced bulk operations.</p>
                
                <form id="bulk-operations-form">
                    <?php wp_nonce_field('geo_local_seo_premium_nonce', 'bulk_nonce'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">Service</th>
                            <td>
                                <input type="text" name="bulk_service" id="bulk-service" class="regular-text" placeholder="e.g., SEO, Plumbing, Marketing" required />
                                <p class="description">The service to generate pages for</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Country Filter</th>
                            <td>
                                <select name="bulk_country" id="bulk-country" class="regular-text">
                                    <option value="">All Countries</option>
                                    <!-- Countries will be loaded dynamically -->
                                </select>
                                <p class="description">Filter locations by country</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Province/State Filter</th>
                            <td>
                                <select name="bulk_province" id="bulk-province" class="regular-text">
                                    <option value="">All Provinces/States</option>
                                    <!-- Provinces will be loaded dynamically -->
                                </select>
                                <p class="description">Filter locations by province/state</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Select Locations</th>
                            <td>
                                <div style="margin-bottom:10px;">
                                    <button type="button" id="select-all-visible-bulk" class="button button-secondary" style="margin-right:10px;">Select All Visible</button>
                                    <button type="button" id="clear-selection-bulk" class="button button-secondary">Clear Selection</button>
                                    <span id="selected-count-bulk" style="margin-left:15px;font-weight:bold;color:#0073aa;">0 locations selected</span>
                                </div>
                                <select name="bulk_locations[]" id="bulk-locations" multiple class="regular-text" style="height: 120px;" required>
                                    <option value="">Loading locations...</option>
                                </select>
                                <p class="description">Use filters above to narrow down selection, then click "Select All Visible" to select all filtered locations.</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Template</th>
                            <td>
                                <select name="bulk_template" id="bulk-template" class="regular-text">
                                    <option value="">Loading templates...</option>
                                </select>
                                <p class="description">Choose a template for all pages</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">URL Structure</th>
                            <td>
                                <select name="bulk_url_logic" id="bulk-url-logic" class="regular-text">
                                    <option value="service_in_location">Service in Location (e.g., "SEO in Amsterdam")</option>
                                    <option value="service_location">Service Location (e.g., "SEO Amsterdam")</option>
                                    <option value="location_service">Location Service (e.g., "Amsterdam SEO")</option>
                                    <option value="service_in_location_province">Service in Location, Province (e.g., "SEO in Amsterdam, North Holland")</option>
                                    <option value="service_in_location_country">Service in Location, Country (e.g., "SEO in Amsterdam, Netherlands")</option>
                                    <option value="custom">🎨 Custom Template (volledig zelf instelbaar)</option>
                                </select>
                                <p class="description">Choose how the page URLs and titles should be structured</p>
                                <div id="bulk-slug-preview" style="background: #f0f0f1; padding: 10px; margin-top: 10px; border-radius: 4px;">
                                    <strong>Voorbeeld URL:</strong> <span id="bulk-slug-example">seo-diensten-in-amsterdam</span>
                                </div>
                            </td>
                        </tr>
                        <tr id="bulk-custom-slug-row" style="display: none;">
                            <th scope="row">Custom Slug Template</th>
                            <td>
                                <input type="text" name="bulk_custom_slug_template" id="bulk-custom-slug-template" class="regular-text" placeholder="[service_name]-in-[location_name]" />
                                <p class="description">
                                    <strong>Beschikbare variabelen:</strong><br>
                                    • <code>[service_name]</code> - Je dienst naam<br>
                                    • <code>[location_name]</code> - Locatie naam<br>
                                    • <code>[province_name]</code> - Provincie naam<br>
                                    • <code>[business_name]</code> - Bedrijfsnaam<br>
                                    • <code>[country]</code> - Land naam<br>
                                    <strong>Voorbeelden:</strong><br>
                                    • <code>[service_name]-[location_name]</code> → "seo-diensten-amsterdam"<br>
                                    • <code>[location_name]-[service_name]</code> → "amsterdam-seo-diensten"<br>
                                    • <code>[service_name]-diensten-[location_name]</code> → "seo-diensten-amsterdam"<br>
                                    • <code>professionele-[service_name]-in-[location_name]</code> → "professionele-seo-diensten-in-amsterdam"
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Max Pages</th>
                            <td>
                                <input type="number" name="bulk_max_pages" id="bulk-max-pages" value="50" min="1" max="1000" class="small-text" />
                                <p class="description">Maximum number of pages to generate (safety limit)</p>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <input type="submit" class="button-primary" value="Bulk Generate Pages" />
                        <button type="button" id="bulk-seo-optimize" class="button button-secondary">SEO Optimize Existing</button>
                        <span id="bulk-status"></span>
                    </p>
                </form>
                
                <div id="bulk-result"></div>
                
                <!-- Existing Pages Management -->
                <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd;">
                    <h4>📄 Manage Existing Pages</h4>
                    <button type="button" id="load-existing-pages" class="button">Load Existing Pages</button>
                    <div id="existing-pages-list" style="margin-top: 15px;"></div>
                </div>
            </div>
            <?php else: ?>
            <!-- Bulk Operations (Premium - License Required) -->
            <div style="background: #fff3cd; padding: 20px; border: 1px solid #ffeaa7; border-radius: 8px; margin: 20px 0;">
                <h3>⚡ Bulk Operations <span style="background: #dc3545; color: white; padding: 2px 8px; border-radius: 4px; font-size: 12px;">PREMIUM</span></h3>
                <p>Generate multiple pages at once with advanced bulk operations.</p>
                <div style="background: white; padding: 15px; border-radius: 6px; margin-top: 15px;">
                    <p><strong>🔒 Premium Feature</strong></p>
                    <p>This feature requires a valid premium license. Please:</p>
                    <ol>
                        <li>Install the Premium Addon plugin</li>
                        <li>Activate your license in the <a href="<?php echo admin_url('admin.php?page=seo-lead-pages-premium-license'); ?>">Premium License</a> tab</li>
                    </ol>
                    <p><a href="<?php echo admin_url('admin.php?page=seo-lead-pages-premium-info'); ?>" class="button button-primary">View Premium Features</a></p>
                </div>
            </div>
            <?php endif; ?>
            
        
        <script>
        jQuery(document).ready(function($) {
            // Load saved locations and templates
            loadSavedLocations();
            loadSavedTemplates();
            
            // Quick generation form
            $('#quick-generate-form').submit(function(e) {
                e.preventDefault();
                
                var formData = new FormData(this);
                formData.append('action', 'quick_generate_pages');
                
                // Debug: Log form data
                console.log('Form Data:');
                for (var pair of formData.entries()) {
                    console.log(pair[0] + ': ' + pair[1]);
                }
                
                $('#generation-status').html('<span style="color: #0073aa;">Generating pages...</span>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        console.log('AJAX Response:', response);
                        if (response.success) {
                            $('#quick-generation-result').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                            $('#generation-status').html('');
                        } else {
                            var errorMessage = response.data.message || response.data;
                            var errorHtml = '<div class="notice notice-error"><p>' + errorMessage + '</p>';
                            
                            // Check if we need to redirect to shortcode settings
                            if (response.data.redirect_url) {
                                errorHtml += '<p><a href="' + response.data.redirect_url + '" class="button button-primary">Configureer Shortcode Instellingen</a></p>';
                            }
                            
                            errorHtml += '</div>';
                            $('#quick-generation-result').html(errorHtml);
                            $('#generation-status').html('');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('AJAX Error:', error);
                        $('#quick-generation-result').html('<div class="notice notice-error"><p>Error generating pages: ' + error + '</p></div>');
                        $('#generation-status').html('');
                    }
                });
            });
            
            
            function loadSavedLocations() {
                console.log('Loading saved locations...');
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_saved_locations_for_generator',
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        console.log('AJAX Response:', response);
                        if (response.success) {
                            $('#location-select').html(response.data);
                            console.log('Locations loaded successfully');
                        } else {
                            console.log('Error loading locations:', response.data);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('AJAX Error loading locations:', error);
                        $('#location-select').html('<option value="">Error loading locations</option>');
                    }
                });
            }
            
            function loadSavedTemplates() {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_saved_templates_for_generator',
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#template-select').html(response.data);
                        } else {
                            console.log('Error loading templates:', response.data);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('AJAX Error loading templates:', error);
                        $('#template-select').html('<option value="">Error loading templates</option>');
                    }
                });
            }
            
            <?php if ($this->get_premium_active()): ?>
            // Premium Bulk Operations functionality
            loadBulkTemplates();
            loadBulkLocations();
            loadBulkCountries();
            
            // Country filter functionality
            $('#bulk-country').on('change', function() {
                var selectedCountry = $(this).val();
                var $locations = $('#bulk-locations option');
                var $provinceSelect = $('#bulk-province');
                
                // Clear province options
                $provinceSelect.html('<option value="">All Provinces/States</option>');
                
                // Filter locations by country
                $locations.each(function() {
                    var $option = $(this);
                    var country = $option.data('country');
                    
                    if (selectedCountry === '' || country === selectedCountry) {
                        $option.show();
                        
                        // Add province to province filter if not already added
                        var province = $option.data('province');
                        if (province && $provinceSelect.find('option[value="' + province + '"]').length === 0) {
                            $provinceSelect.append('<option value="' + province + '">' + province + '</option>');
                        }
                    } else {
                        $option.hide();
                    }
                });
                
                // Clear location selection when country changes
                $('#bulk-locations').val([]);
                updateBulkSelectedCount();
            });
            
            // Province filter functionality
            $('#bulk-province').on('change', function() {
                var selectedProvince = $(this).val();
                var selectedCountry = $('#bulk-country').val();
                var $locations = $('#bulk-locations option');
                
                $locations.each(function() {
                    var $option = $(this);
                    var country = $option.data('country');
                    var province = $option.data('province');
                    
                    var showOption = true;
                    
                    // Apply country filter
                    if (selectedCountry !== '' && country !== selectedCountry) {
                        showOption = false;
                    }
                    
                    // Apply province filter
                    if (selectedProvince !== '' && province !== selectedProvince) {
                        showOption = false;
                    }
                    
                    if (showOption) {
                        $option.show();
                    } else {
                        $option.hide();
                    }
                });
                
                // Clear location selection when province changes
                $('#bulk-locations').val([]);
                updateBulkSelectedCount();
            });
            
            // Update selected count display for bulk operations
            function updateBulkSelectedCount() {
                var selectedCount = $('#bulk-locations').val() ? $('#bulk-locations').val().length : 0;
                $('#selected-count-bulk').text(selectedCount + ' locations selected');
            }
            
            // Select all visible locations for bulk operations
            $('#select-all-visible-bulk').on('click', function() {
                var $locations = $('#bulk-locations');
                var visibleValues = [];
                
                $locations.find('option:visible').each(function() {
                    if ($(this).val() !== '') { // Skip empty options
                        visibleValues.push($(this).val());
                    }
                });
                
                $locations.val(visibleValues);
                updateBulkSelectedCount();
            });
            
            // Clear all selections for bulk operations
            $('#clear-selection-bulk').on('click', function() {
                $('#bulk-locations').val([]);
                updateBulkSelectedCount();
            });
            
            // Update count when selection changes for bulk operations
            $('#bulk-locations').on('change', updateBulkSelectedCount);
            
            // Bulk operations form
            $('#bulk-operations-form').submit(function(e) {
                e.preventDefault();
                
                var service = $('#bulk-service').val();
                var locations = $('#bulk-locations').val();
                var template = $('#bulk-template').val();
                var maxPages = $('#bulk-max-pages').val();
                
                if (!service || !locations || locations.length === 0) {
                    alert('Please fill in service and select at least one location.');
                    return;
                }
                
                if (confirm('This will generate pages for ' + locations.length + ' locations. Continue?')) {
                    $('#bulk-status').html('Generating pages...');
                    $('#bulk-result').html('<div class="notice notice-info"><p>Generating pages... Please wait.</p></div>');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'geo_local_seo_premium_bulk_generate_pages',
                            service: service,
                            locations: locations,
                            template: template,
                            max_pages: maxPages,
                            url_logic: $('#bulk-url-logic').val(),
                            country: $('#bulk-country').val(),
                            province: $('#bulk-province').val(),
                            nonce: '<?php echo wp_create_nonce('geo_local_seo_premium_nonce'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                $('#bulk-result').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                                if (response.data.errors && response.data.errors.length > 0) {
                                    $('#bulk-result').append('<div class="notice notice-warning"><p>Errors: ' + response.data.errors.join(', ') + '</p></div>');
                                }
                            } else {
                                $('#bulk-result').html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>');
                            }
                            $('#bulk-status').html('');
                        },
                        error: function() {
                            $('#bulk-result').html('<div class="notice notice-error"><p>Error generating pages.</p></div>');
                            $('#bulk-status').html('');
                        }
                    });
                }
            });
            
            // SEO Optimize button
            $('#bulk-seo-optimize').click(function() {
                var selectedPages = [];
                $('input[name="page_ids[]"]:checked').each(function() {
                    selectedPages.push($(this).val());
                });
                
                if (selectedPages.length === 0) {
                    alert('Please select pages to optimize first.');
                    return;
                }
                
                if (confirm('SEO optimize ' + selectedPages.length + ' selected pages?')) {
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'geo_local_seo_premium_bulk_seo_optimize',
                            page_ids: selectedPages,
                            nonce: '<?php echo wp_create_nonce('geo_local_seo_premium_nonce'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                $('#bulk-result').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                            } else {
                                $('#bulk-result').html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>');
                            }
                        },
                        error: function() {
                            $('#bulk-result').html('<div class="notice notice-error"><p>Error optimizing pages.</p></div>');
                        }
                    });
                }
            });
            
            // Load existing pages
            $('#load-existing-pages').click(function() {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'geo_local_seo_premium_get_pages',
                        nonce: '<?php echo wp_create_nonce('geo_local_seo_premium_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            var html = '<table class="wp-list-table widefat fixed striped"><thead><tr><th><input type="checkbox" id="select-all-pages"></th><th>Page Title</th><th>Service</th><th>Location</th><th>Actions</th></tr></thead><tbody>';
                            
                            response.data.pages.forEach(function(page) {
                                html += '<tr>';
                                html += '<td><input type="checkbox" name="page_ids[]" value="' + page.ID + '"></td>';
                                html += '<td>' + page.post_title + '</td>';
                                html += '<td>' + (page.service || 'N/A') + '</td>';
                                html += '<td>' + (page.location || 'N/A') + '</td>';
                                html += '<td><a href="' + '<?php echo admin_url('post.php?post='); ?>' + page.ID + '&action=edit" class="button button-small">Edit</a></td>';
                                html += '</tr>';
                            });
                            
                            html += '</tbody></table>';
                            $('#existing-pages-list').html(html);
                            
                            // Select all functionality
                            $('#select-all-pages').change(function() {
                                $('input[name="page_ids[]"]').prop('checked', this.checked);
                            });
                        } else {
                            $('#existing-pages-list').html('<p>No pages found.</p>');
                        }
                    },
                    error: function() {
                        $('#existing-pages-list').html('<p>Error loading pages.</p>');
                    }
                });
            });
            
            function loadBulkTemplates() {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'geo_local_seo_premium_get_templates',
                        nonce: '<?php echo wp_create_nonce('geo_local_seo_premium_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            var select = $('#bulk-template');
                            select.empty();
                            select.append('<option value="">Use default template</option>');
                            
                            response.data.templates.forEach(function(template) {
                                select.append('<option value="' + template.value + '">' + template.label + ' (' + template.type + ')</option>');
                            });
                        }
                    }
                });
            }
            
            function loadBulkCountries() {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'geo_local_seo_premium_get_countries',
                        nonce: '<?php echo wp_create_nonce('geo_local_seo_premium_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            var select = $('#bulk-country');
                            select.empty();
                            select.append('<option value="">All Countries</option>');
                            
                            response.data.countries.forEach(function(country) {
                                select.append('<option value="' + country.name + '">' + country.name + '</option>');
                            });
                        }
                    }
                });
            }
            
            function loadBulkLocations() {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'geo_local_seo_premium_get_locations',
                        nonce: '<?php echo wp_create_nonce('geo_local_seo_premium_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            var select = $('#bulk-locations');
                            select.empty();
                            select.append('<option value="">Loading locations...</option>');
                            
                            response.data.locations.forEach(function(location) {
                                select.append('<option value="' + location.value + '" data-country="' + location.country + '" data-province="' + location.province + '">' + location.label + '</option>');
                            });
                        }
                    }
                });
            }
            <?php endif; ?>
            
            // Slug preview functionality
            function updateSlugPreview() {
                var serviceName = 'SEO Diensten'; // Default service name
                var slugTemplate = $('#url-logic').val() || 'service_in_location';
                var locationName = 'Amsterdam'; // Example location
                
                var slug = '';
                if (slugTemplate === 'custom') {
                    var customTemplate = $('#custom-slug-template').val() || '[service_name]-in-[location_name]';
                    slug = customTemplate
                        .replace(/\[service_name\]/g, serviceName.toLowerCase().replace(/\s+/g, '-'))
                        .replace(/\[location_name\]/g, locationName.toLowerCase())
                        .replace(/\[province_name\]/g, 'noord-holland')
                        .replace(/\[business_name\]/g, 'uw-bedrijf')
                        .replace(/\[country\]/g, 'nederland');
                } else {
                    switch(slugTemplate) {
                        case 'service_location':
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-' + locationName.toLowerCase();
                            break;
                        case 'service_in_location':
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-in-' + locationName.toLowerCase();
                            break;
                        case 'location_service':
                            slug = locationName.toLowerCase() + '-' + serviceName.toLowerCase().replace(/\s+/g, '-');
                            break;
                        case 'service_in_location_province':
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-in-' + locationName.toLowerCase() + '-noord-holland';
                            break;
                        case 'service_in_location_country':
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-in-' + locationName.toLowerCase() + '-nederland';
                            break;
                        default:
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-in-' + locationName.toLowerCase();
                    }
                }
                
                $('#slug-example').text(slug);
            }
            
            function updateBulkSlugPreview() {
                var serviceName = 'SEO Diensten'; // Default service name
                var slugTemplate = $('#bulk-url-logic').val() || 'service_in_location';
                var locationName = 'Amsterdam'; // Example location
                
                var slug = '';
                if (slugTemplate === 'custom') {
                    var customTemplate = $('#bulk-custom-slug-template').val() || '[service_name]-in-[location_name]';
                    slug = customTemplate
                        .replace(/\[service_name\]/g, serviceName.toLowerCase().replace(/\s+/g, '-'))
                        .replace(/\[location_name\]/g, locationName.toLowerCase())
                        .replace(/\[province_name\]/g, 'noord-holland')
                        .replace(/\[business_name\]/g, 'uw-bedrijf')
                        .replace(/\[country\]/g, 'nederland');
                } else {
                    switch(slugTemplate) {
                        case 'service_location':
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-' + locationName.toLowerCase();
                            break;
                        case 'service_in_location':
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-in-' + locationName.toLowerCase();
                            break;
                        case 'location_service':
                            slug = locationName.toLowerCase() + '-' + serviceName.toLowerCase().replace(/\s+/g, '-');
                            break;
                        case 'service_in_location_province':
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-in-' + locationName.toLowerCase() + '-noord-holland';
                            break;
                        case 'service_in_location_country':
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-in-' + locationName.toLowerCase() + '-nederland';
                            break;
                        default:
                            slug = serviceName.toLowerCase().replace(/\s+/g, '-') + '-in-' + locationName.toLowerCase();
                    }
                }
                
                $('#bulk-slug-example').text(slug);
            }
            
            function toggleCustomSlugRow() {
                var slugTemplate = $('#url-logic').val();
                if (slugTemplate === 'custom') {
                    $('#custom-slug-row').show();
                } else {
                    $('#custom-slug-row').hide();
                }
            }
            
            function toggleBulkCustomSlugRow() {
                var slugTemplate = $('#bulk-url-logic').val();
                if (slugTemplate === 'custom') {
                    $('#bulk-custom-slug-row').show();
                } else {
                    $('#bulk-custom-slug-row').hide();
                }
            }
            
            // Update preview when inputs change
            $('#url-logic, #custom-slug-template').on('input change', function() {
                updateSlugPreview();
                toggleCustomSlugRow();
            });
            
            $('#bulk-url-logic, #bulk-custom-slug-template').on('input change', function() {
                updateBulkSlugPreview();
                toggleBulkCustomSlugRow();
            });
            
            // Initial update
            updateSlugPreview();
            toggleCustomSlugRow();
            updateBulkSlugPreview();
            toggleBulkCustomSlugRow();
        });
        </script>
        <?php
    }
    
    public function locations_manager_page() {
        // Ensure premium status is checked when page is loaded
        $this->check_premium_status();
        $this->premium_addon_notice();
        ?>
        <div class="wrap">
            <h1>Locations Manager</h1>
            <p>Manage your business locations and generate pages for multiple locations.</p>
            
            <!-- Add Location Manually -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>📍 Add Location Manually</h3>
                <form id="add-location-form" onsubmit="return false;">
                    <?php wp_nonce_field('seo_lead_pages_nonce', 'nonce'); ?>
                    <script>
                    // Function to attach delete handlers
                    function attachDeleteHandlers() {
                        // Handle delete saved location buttons
                        document.querySelectorAll('.delete-location').forEach(function(button) {
                            button.addEventListener('click', function(e) {
                                e.preventDefault();
                                var locationIndex = this.getAttribute('data-location-index');
                                
                                if (confirm('Are you sure you want to delete this location?')) {
                                    var originalText = this.textContent;
                                    this.textContent = 'Deleting...';
                                    this.disabled = true;
                                    
                                    fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                        method: 'POST',
                                        headers: {
                                            'Content-Type': 'application/x-www-form-urlencoded',
                                        },
                                        body: 'action=delete_saved_location&location_index=' + locationIndex + '&nonce=<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                                    })
                                    .then(response => response.json())
                                    .then(data => {
                                        if (data.success) {
                                            // Reload the locations list
                                            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                                method: 'POST',
                                                headers: {
                                                    'Content-Type': 'application/x-www-form-urlencoded',
                                                },
                                                body: 'action=get_saved_locations&nonce=<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                                            })
                                            .then(response => response.json())
                                            .then(data => {
                                                if (data.success) {
                                                    var savedLocationsDiv = document.querySelector('#saved-locations-list');
                                                    if (savedLocationsDiv && data.data) {
                                                        savedLocationsDiv.innerHTML = data.data;
                                                        attachDeleteHandlers(); // Re-attach handlers
                                                    }
                                                }
                                            })
                                            .catch(error => {
                                                console.error('Error reloading locations after delete:', error);
                                                window.location.reload();
                                            });
                                        } else {
                                            alert('Error deleting location: ' + (data.data || 'Unknown error'));
                                        }
                                    })
                                    .catch(error => {
                                        console.error('Error deleting location:', error);
                                        alert('Error deleting location: ' + error.message);
                                    })
                                    .finally(() => {
                                        this.textContent = originalText;
                                        this.disabled = false;
                                    });
                                }
                            });
                        });
                        
                        // Handle delete imported location buttons
                        document.querySelectorAll('.delete-imported-location').forEach(function(button) {
                            button.addEventListener('click', function(e) {
                                e.preventDefault();
                                var locationId = this.getAttribute('data-location-id');
                                
                                if (confirm('Are you sure you want to delete this imported location?')) {
                                    var originalText = this.textContent;
                                    this.textContent = 'Deleting...';
                                    this.disabled = true;
                                    
                                    fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                        method: 'POST',
                                        headers: {
                                            'Content-Type': 'application/x-www-form-urlencoded',
                                        },
                                        body: 'action=delete_imported_location&location_id=' + locationId + '&nonce=<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                                    })
                                    .then(response => response.json())
                                    .then(data => {
                                        if (data.success) {
                                            // Reload the locations list
                                            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                                method: 'POST',
                                                headers: {
                                                    'Content-Type': 'application/x-www-form-urlencoded',
                                                },
                                                body: 'action=get_saved_locations&nonce=<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                                            })
                                            .then(response => response.json())
                                            .then(data => {
                                                if (data.success) {
                                                    var savedLocationsDiv = document.querySelector('#saved-locations-list');
                                                    if (savedLocationsDiv && data.data) {
                                                        savedLocationsDiv.innerHTML = data.data;
                                                        attachDeleteHandlers(); // Re-attach handlers
                                                    }
                                                }
                                            })
                                            .catch(error => {
                                                console.error('Error reloading locations after delete:', error);
                                                window.location.reload();
                                            });
                                        } else {
                                            alert('Error deleting imported location: ' + (data.data || 'Unknown error'));
                                        }
                                    })
                                    .catch(error => {
                                        console.error('Error deleting imported location:', error);
                                        alert('Error deleting imported location: ' + error.message);
                                    })
                                    .finally(() => {
                                        this.textContent = originalText;
                                        this.disabled = false;
                                    });
                                }
                            });
                        });
                    }
                    
                    // Inline event handler with AJAX functionality
                    document.addEventListener('DOMContentLoaded', function() {
                        // Attach delete handlers on page load
                        attachDeleteHandlers();
                        var submitBtn = document.querySelector('#add-location-form input[type="submit"]');
                        if (submitBtn) {
                            submitBtn.addEventListener('click', function(e) {
                                e.preventDefault();
                                console.log('Button clicked via inline handler - starting AJAX call');
                                
                                // Get form data
                                var form = document.getElementById('add-location-form');
                                var formData = new FormData(form);
                                formData.append('action', 'add_location_manually');
                                
                                // Ensure nonce is fresh
                                var freshNonce = '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>';
                                formData.set('nonce', freshNonce);
                                
                                // Show loading state
                                var originalText = submitBtn.value;
                                submitBtn.value = 'Adding...';
                                submitBtn.disabled = true;
                                
                                // Make AJAX call
                                fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                    method: 'POST',
                                    body: formData
                                })
                                .then(response => response.json())
                                .then(data => {
                                    console.log('AJAX response:', data);
                                    if (data.success) {
                                        // Show success message
                                        var resultDiv = document.getElementById('add-location-result');
                                        if (resultDiv) {
                                            resultDiv.innerHTML = '<div class="notice notice-success"><p>Location added successfully!</p></div>';
                                        }
                                        form.reset();
                                        // Reload locations list by making AJAX call
                                        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                                            method: 'POST',
                                            headers: {
                                                'Content-Type': 'application/x-www-form-urlencoded',
                                            },
                                            body: 'action=get_saved_locations&nonce=<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                                        })
                                        .then(response => response.json())
                                        .then(data => {
                                            if (data.success) {
                                                // Update the saved locations list
                                                var savedLocationsDiv = document.querySelector('#saved-locations-list');
                                                if (savedLocationsDiv && data.data) {
                                                    savedLocationsDiv.innerHTML = data.data;
                                                    
                                                    // Re-attach event handlers for delete buttons
                                                    attachDeleteHandlers();
                                                }
                                            }
                                        })
                                        .catch(error => {
                                            console.error('Error reloading locations:', error);
                                            // Fallback: refresh the page to show updated locations
                                            setTimeout(function() {
                                                window.location.reload();
                                            }, 1000);
                                        });
                                    } else {
                                        // Show error message
                                        var resultDiv = document.getElementById('add-location-result');
                                        if (resultDiv) {
                                            resultDiv.innerHTML = '<div class="notice notice-error"><p>Error: ' + (data.data || 'Unknown error') + '</p></div>';
                                        }
                                    }
                                })
                                .catch(error => {
                                    console.error('AJAX error:', error);
                                    // Show error message in UI
                                    var resultDiv = document.getElementById('add-location-result');
                                    if (resultDiv) {
                                        resultDiv.innerHTML = '<div class="notice notice-error"><p>Error adding location: ' + error.message + '</p></div>';
                                    }
                                })
                                .finally(() => {
                                    submitBtn.value = originalText;
                                    submitBtn.disabled = false;
                                });
                                
                                return false;
                            });
                        }
                    });
                    </script>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">City Name *</th>
                            <td><input type="text" name="city_name" class="regular-text" required placeholder="e.g. Amsterdam" /></td>
                        </tr>
                        <tr>
                            <th scope="row">Province/State *</th>
                            <td><input type="text" name="province_state" class="regular-text" required placeholder="e.g. North Holland" /></td>
                        </tr>
                        <tr>
                            <th scope="row">Country *</th>
                            <td>
                                <select name="country" id="country-select" class="regular-text" required>
                                    <option value="">Select Country</option>
                                    <option value="Netherlands">Netherlands</option>
                                    <option value="Belgium">Belgium</option>
                                    <option value="Germany">Germany</option>
                                    <option value="France">France</option>
                                    <option value="United Kingdom">United Kingdom</option>
                                    <option value="Spain">Spain</option>
                                    <option value="Italy">Italy</option>
                                    <option value="United States">United States</option>
                                    <option value="Canada">Canada</option>
                                    <option value="Australia">Australia</option>
                                    <option value="Austria">Austria</option>
                                    <option value="Switzerland">Switzerland</option>
                                    <option value="Denmark">Denmark</option>
                                    <option value="Sweden">Sweden</option>
                                    <option value="Norway">Norway</option>
                                    <option value="Finland">Finland</option>
                                    <option value="Poland">Poland</option>
                                    <option value="Czech Republic">Czech Republic</option>
                                    <option value="Hungary">Hungary</option>
                                    <option value="Portugal">Portugal</option>
                                    <option value="Greece">Greece</option>
                                    <option value="Ireland">Ireland</option>
                                    <option value="Luxembourg">Luxembourg</option>
                                    <option value="New Zealand">New Zealand</option>
                                    <option value="South Africa">South Africa</option>
                                    <option value="Brazil">Brazil</option>
                                    <option value="Mexico">Mexico</option>
                                    <option value="Japan">Japan</option>
                                    <option value="South Korea">South Korea</option>
                                    <option value="Singapore">Singapore</option>
                                    <option value="India">India</option>
                                    <option value="China">China</option>
                                    <option value="Thailand">Thailand</option>
                                    <option value="Malaysia">Malaysia</option>
                                    <option value="Indonesia">Indonesia</option>
                                    <option value="Philippines">Philippines</option>
                                    <option value="Vietnam">Vietnam</option>
                                    <option value="Turkey">Turkey</option>
                                    <option value="Israel">Israel</option>
                                    <option value="United Arab Emirates">United Arab Emirates</option>
                                    <option value="Saudi Arabia">Saudi Arabia</option>
                                    <option value="Egypt">Egypt</option>
                                    <option value="Morocco">Morocco</option>
                                    <option value="Nigeria">Nigeria</option>
                                    <option value="Kenya">Kenya</option>
                                    <option value="Ghana">Ghana</option>
                                    <option value="Argentina">Argentina</option>
                                    <option value="Chile">Chile</option>
                                    <option value="Colombia">Colombia</option>
                                    <option value="Peru">Peru</option>
                                    <option value="Venezuela">Venezuela</option>
                                    <option value="Russia">Russia</option>
                                    <option value="Ukraine">Ukraine</option>
                                    <option value="Romania">Romania</option>
                                    <option value="Bulgaria">Bulgaria</option>
                                    <option value="Croatia">Croatia</option>
                                    <option value="Slovenia">Slovenia</option>
                                    <option value="Slovakia">Slovakia</option>
                                    <option value="Estonia">Estonia</option>
                                    <option value="Latvia">Latvia</option>
                                    <option value="Lithuania">Lithuania</option>
                                    <option value="custom">+ Add Custom Country</option>
                                </select>
                                <div id="custom-country-input" style="display: none; margin-top: 10px;">
                                    <input type="text" name="custom_country" class="regular-text" placeholder="Enter country name" />
                                    <p class="description">Enter the name of the country you want to add.</p>
                                </div>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Postal Code</th>
                            <td><input type="text" name="postal_code" class="regular-text" placeholder="e.g. 1012 AB" /></td>
                        </tr>
                        <tr>
                            <th scope="row">Business Name *</th>
                            <td><input type="text" name="business_name" class="regular-text" required placeholder="e.g. Your Business Name" /></td>
                        </tr>
                        <tr>
                            <th scope="row">Service *</th>
                            <td><input type="text" name="service" class="regular-text" required placeholder="e.g. Plumbing Services" /></td>
                        </tr>
                        <tr>
                            <th scope="row">Phone Number</th>
                            <td><input type="text" name="phone" class="regular-text" placeholder="e.g. +31 20 123 4567" /></td>
                        </tr>
                        <tr>
                            <th scope="row">Email</th>
                            <td><input type="email" name="email" class="regular-text" placeholder="e.g. info@yourbusiness.com" /></td>
                        </tr>
                        <tr>
                            <th scope="row">Website</th>
                            <td><input type="url" name="website" class="regular-text" placeholder="e.g. https://yourbusiness.com" /></td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <input type="submit" class="button-primary" value="Add Location" />
                        <button type="button" id="generate-page-for-location" class="button button-secondary" style="margin-left: 10px;">Add Location & Generate Page</button>
                    </p>
                </form>
                
                <div id="add-location-result"></div>
            </div>
            
            <!-- Import Locations from CSV -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>📁 Import Locations from CSV</h3>
                <p>Upload a CSV file with your locations. Format: business_name,location,service,phone,email,website</p>
                
                <form id="import-locations-form">
                    <?php wp_nonce_field('seo_lead_pages_nonce', 'nonce'); ?>
                    
                    <input type="file" name="csv_file" accept=".csv" required />
                    <p class="submit">
                        <input type="submit" class="button-primary" value="Import Locations" />
                    </p>
                </form>
                
                <div id="import-result"></div>
            </div>
            
            <!-- Bulk Delete Locations -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>🗑️ Bulk Delete Locations</h3>
                <p>Delete multiple locations at once by country or province/state.</p>
                
                <form id="bulk-delete-form">
                    <?php wp_nonce_field('seo_lead_pages_nonce', 'nonce'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">Delete by Country</th>
                            <td>
                                <select name="bulk_delete_country" id="bulk-delete-country" class="regular-text">
                                    <option value="">Select Country</option>
                                    <!-- Countries will be loaded dynamically -->
                                </select>
                                <p class="description">Delete all locations from a specific country</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Delete by Province/State</th>
                            <td>
                                <select name="bulk_delete_province" id="bulk-delete-province" class="regular-text">
                                    <option value="">Select Province/State</option>
                                    <!-- Provinces will be loaded dynamically -->
                                </select>
                                <p class="description">Delete all locations from a specific province/state</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Delete Type</th>
                            <td>
                                <label>
                                    <input type="radio" name="delete_type" value="imported" checked /> Imported Locations Only
                                </label><br>
                                <label>
                                    <input type="radio" name="delete_type" value="manual" /> Manually Added Locations Only
                                </label><br>
                                <label>
                                    <input type="radio" name="delete_type" value="both" /> Both Imported and Manual
                                </label>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <button type="button" id="bulk-delete-btn" class="button button-secondary" style="background-color: #dc3232; color: white; border-color: #dc3232;">
                            🗑️ Delete Selected Locations
                        </button>
                        <span id="bulk-delete-count" style="margin-left: 10px; font-weight: bold; color: #dc3232;"></span>
                    </p>
                </form>
                
                <div id="bulk-delete-result"></div>
            </div>
            
            <!-- Bulk Delete Saved Locations -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>🗑️ Bulk Delete Saved Locations</h3>
                <p>Delete multiple saved locations at once by country or province.</p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Filter by Country</th>
                        <td>
                            <select id="saved-locations-country" class="regular-text">
                                <option value="">All Countries</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Filter by Province/State</th>
                        <td>
                            <select id="saved-locations-province" class="regular-text">
                                <option value="">All Provinces/States</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Selected Locations</th>
                        <td>
                            <div style="margin-bottom: 10px;">
                                <button type="button" id="select-all-saved-locations" class="button button-secondary" style="margin-right: 10px;">Select All Visible</button>
                                <button type="button" id="clear-saved-locations-selection" class="button button-secondary" style="margin-right: 10px;">Clear Selection</button>
                                <span id="saved-locations-selected-count" style="font-weight: bold; color: #0073aa;">0 locations selected</span>
                            </div>
                            <select id="saved-locations-select" multiple class="regular-text" style="height: 120px;">
                                <option value="">Loading locations...</option>
                            </select>
                            <p class="description">Use filters above to narrow down selection, then click "Select All Visible" to select all filtered locations.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Actions</th>
                        <td>
                            <button type="button" id="bulk-delete-saved-locations" class="button button-primary" style="background: #d63638; border-color: #d63638;">Delete Selected Locations</button>
                            <div id="saved-locations-bulk-result" style="margin-top: 10px;"></div>
                        </td>
                    </tr>
                </table>
            </div>

            <!-- Saved Locations -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>💾 Saved Locations</h3>
                <div id="saved-locations-list">
                    <!-- Locations will be loaded here via AJAX -->
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            try {
                console.log('[GEO SEO DEBUG] JavaScript loaded and jQuery ready for Locations Manager');
            
            // Debug logging function
            function debugLog(action, message, data) {
                console.log('[GEO SEO DEBUG] ' + action + ': ' + message);
                if (data) {
                    console.log('[GEO SEO DEBUG] Data:', data);
                }
            }
            
            // Check if form exists
            if ($('#add-location-form').length === 0) {
                console.error('[GEO SEO DEBUG] ERROR: add-location-form not found!');
            } else {
                console.log('[GEO SEO DEBUG] add-location-form found, setting up event handlers');
            }
            
            // Ensure ajaxurl is defined
            if (typeof ajaxurl === 'undefined') {
                console.error('[GEO SEO DEBUG] ERROR: ajaxurl is not defined!');
                window.ajaxurl = '<?php echo admin_url('admin-ajax.php'); ?>';
                console.log('[GEO SEO DEBUG] ajaxurl manually set to:', window.ajaxurl);
            }
            
            // Load saved locations
            loadSavedLocations();
            
            // Load bulk delete options
            loadBulkDeleteOptions();
            
            // Load saved locations for bulk delete
            loadSavedLocationsForBulkDelete();
            
            // Handle custom country selection
            $('#country-select').change(function() {
                if ($(this).val() === 'custom') {
                    $('#custom-country-input').show();
                    $('input[name="custom_country"]').focus();
                } else {
                    $('#custom-country-input').hide();
                    $('input[name="custom_country"]').val('');
                }
            });
            
            // Add location form - use event delegation to ensure it works
            console.log('[GEO SEO DEBUG] Setting up add-location-form submit handler');
            
            // Also handle submit button clicks directly
            $(document).on('click', '#add-location-form input[type="submit"]', function(e) {
                e.preventDefault();
                e.stopPropagation();
                console.log('[GEO SEO DEBUG] Submit button clicked directly');
                $('#add-location-form').trigger('submit');
                return false;
            });
            
            // Prevent default form submission and handle via AJAX
            $(document).on('submit', '#add-location-form', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                console.log('[GEO SEO DEBUG] addLocationForm: Form submission started');
                debugLog('addLocationForm', 'Form submission started');
                
                var formData = new FormData(this);
                formData.append('action', 'add_location_manually');
                
                // Ensure nonce is fresh
                var freshNonce = '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>';
                formData.set('nonce', freshNonce);
                
                // Log form data for debugging
                var formDataObj = {};
                for (var pair of formData.entries()) {
                    formDataObj[pair[0]] = pair[1];
                }
                console.log('[GEO SEO DEBUG] addLocationForm: Form data prepared', formDataObj);
                debugLog('addLocationForm', 'Form data prepared', formDataObj);
                
                // Show loading state
                var $submitButton = $(this).find('input[type="submit"]');
                console.log('[GEO SEO DEBUG] Submit button found:', $submitButton.length > 0);
                if ($submitButton.length === 0) {
                    console.error('[GEO SEO DEBUG] ERROR: Submit button not found!');
                    return;
                }
                var originalText = $submitButton.val();
                $submitButton.val('Adding...').prop('disabled', true);
                
                // Ensure ajaxurl is available
                var ajaxUrl = (typeof ajaxurl !== 'undefined') ? ajaxurl : '<?php echo admin_url('admin-ajax.php'); ?>';
                
                console.log('[GEO SEO DEBUG] addLocationForm: Starting AJAX call to', ajaxUrl);
                console.log('[GEO SEO DEBUG] ajaxurl type:', typeof ajaxurl);
                console.log('[GEO SEO DEBUG] Form data being sent:', formDataObj);
                $.ajax({
                    url: ajaxUrl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        console.log('[GEO SEO DEBUG] addLocationForm: AJAX call successful', response);
                        debugLog('addLocationForm', 'AJAX call successful', response);
                        if (response.success) {
                            debugLog('addLocationForm', 'Location added successfully');
                            $('#add-location-result').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                            $('#add-location-form')[0].reset();
                            loadSavedLocations();
                        } else {
                            debugLog('addLocationForm', 'Response indicates failure', response);
                            $('#add-location-result').html('<div class="notice notice-error"><p>' + (response.data || 'An error occurred') + '</p></div>');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('[GEO SEO DEBUG] addLocationForm: AJAX call failed', {status: status, error: error, responseText: xhr.responseText});
                        debugLog('addLocationForm', 'AJAX call failed', {status: status, error: error, responseText: xhr.responseText});
                        console.error('AJAX Error:', error);
                        console.error('Response:', xhr.responseText);
                        $('#add-location-result').html('<div class="notice notice-error"><p>Er is een fout opgetreden bij het toevoegen van de locatie. Controleer de console voor meer details.</p></div>');
                    },
                    complete: function() {
                        $submitButton.val(originalText).prop('disabled', false);
                    }
                });
            });
            
            // Add location and generate page
            $('#generate-page-for-location').click(function(e) {
                e.preventDefault();
                
                debugLog('addLocationAndGenerate', 'Button clicked, starting process');
                
                var formData = new FormData($('#add-location-form')[0]);
                formData.append('action', 'add_location_and_generate_page');
                
                // Ensure nonce is fresh
                var freshNonce = '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>';
                formData.set('nonce', freshNonce);
                
                // Log form data for debugging
                var formDataObj = {};
                for (var pair of formData.entries()) {
                    formDataObj[pair[0]] = pair[1];
                }
                debugLog('addLocationAndGenerate', 'Form data prepared', formDataObj);
                
                // Show loading state
                var $button = $(this);
                var originalText = $button.text();
                $button.text('Adding & Generating...').prop('disabled', true);
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        debugLog('addLocationAndGenerate', 'AJAX call successful', response);
                        if (response.success) {
                            debugLog('addLocationAndGenerate', 'Location added and page generated successfully');
                            $('#add-location-result').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                            $('#add-location-form')[0].reset();
                            loadSavedLocations();
                        } else {
                            debugLog('addLocationAndGenerate', 'Response indicates failure', response);
                            $('#add-location-result').html('<div class="notice notice-error"><p>' + (response.data || 'An error occurred') + '</p></div>');
                        }
                    },
                    error: function(xhr, status, error) {
                        debugLog('addLocationAndGenerate', 'AJAX call failed', {status: status, error: error, responseText: xhr.responseText});
                        console.error('AJAX Error:', error);
                        console.error('Response:', xhr.responseText);
                        $('#add-location-result').html('<div class="notice notice-error"><p>Er is een fout opgetreden bij het toevoegen van de locatie en genereren van de pagina. Controleer de console voor meer details.</p></div>');
                    },
                    complete: function() {
                        $button.text(originalText).prop('disabled', false);
                    }
                });
            });
            
            // Import locations form
            $('#import-locations-form').submit(function(e) {
                e.preventDefault();
                
                var formData = new FormData(this);
                formData.append('action', 'import_locations_csv');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        if (response.success) {
                            $('#import-result').html('<div class="notice notice-success"><p>' + response.data + '</p></div>');
                            loadSavedLocations();
                        } else {
                            $('#import-result').html('<div class="notice notice-error"><p>' + response.data + '</p></div>');
                        }
                    }
                });
            });
            
            function loadSavedLocations() {
                debugLog('loadSavedLocations', 'Starting to load saved locations');
                
                // Ensure ajaxurl is available
                var ajaxUrl = (typeof ajaxurl !== 'undefined') ? ajaxurl : '<?php echo admin_url('admin-ajax.php'); ?>';
                
                $.ajax({
                    url: ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'get_saved_locations',
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        debugLog('loadSavedLocations', 'AJAX call successful', response);
                        
                        if (response.success) {
                            debugLog('loadSavedLocations', 'Response indicates success, updating UI');
                            $('#saved-locations-list').html(response.data);
                            
                            // Add event handlers for dynamically loaded buttons
                            
                            $('.delete-location').off('click').on('click', function() {
                                var locationIndex = $(this).data('location-index');
                                
                                if (confirm('Are you sure you want to delete this location?')) {
                                    $.ajax({
                                        url: ajaxurl,
                                        type: 'POST',
                                        data: {
                                            action: 'delete_saved_location',
                                            location_index: locationIndex,
                                            nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                                        },
                                        success: function(response) {
                                            if (response.success) {
                                                loadSavedLocations(); // Reload the list
                                            } else {
                                                alert('Error deleting location: ' + response.data);
                                            }
                                        },
                                        error: function(xhr, status, error) {
                                            console.error('Error deleting location:', error);
                                            alert('Error deleting location. Please try again.');
                                        }
                                    });
                                }
                            });
                            
                            // Handle delete imported location buttons
                            $('.delete-imported-location').off('click').on('click', function() {
                                var locationId = $(this).data('location-id');
                                
                                if (confirm('Are you sure you want to delete this imported location?')) {
                                    $.ajax({
                                        url: ajaxurl,
                                        type: 'POST',
                                        data: {
                                            action: 'delete_imported_location',
                                            location_id: locationId,
                                            nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                                        },
                                        success: function(response) {
                                            if (response.success) {
                                                loadSavedLocations(); // Reload the list
                                            } else {
                                                alert('Error deleting imported location: ' + response.data);
                                            }
                                        },
                                        error: function(xhr, status, error) {
                                            console.error('Error deleting imported location:', error);
                                            alert('Error deleting imported location. Please try again.');
                                        }
                                    });
                                }
                            });
                        } else {
                            debugLog('loadSavedLocations', 'Response indicates failure', response);
                            console.error('Failed to load saved locations:', response.data);
                            $('#saved-locations-list').html('<div class="notice notice-error"><p>Error loading saved locations: ' + (response.data || 'Unknown error') + '</p></div>');
                        }
                    },
                    error: function(xhr, status, error) {
                        debugLog('loadSavedLocations', 'AJAX call failed', {status: status, error: error, responseText: xhr.responseText});
                        console.error('AJAX Error loading saved locations:', error);
                        console.error('Response:', xhr.responseText);
                        $('#saved-locations-list').html('<div class="notice notice-error"><p>Er is een fout opgetreden bij het laden van de opgeslagen locaties. Controleer de console voor meer details.</p></div>');
                    }
                });
            }
            
            // Load countries functionality for bulk delete (available in free version)
            console.log('GEO Local SEO: Loading countries functionality for bulk delete...');
            loadCountries();
            
            function loadCountries() {
                debugLog('loadCountries', 'Starting to load countries');
                console.log('GEO Local SEO: loadCountries() called');
                
                // Ensure ajaxurl is available
                var ajaxUrl = (typeof ajaxurl !== 'undefined') ? ajaxurl : '<?php echo admin_url('admin-ajax.php'); ?>';
                
                // Free version - use basic countries list for bulk delete functionality
                debugLog('loadCountries', 'Using free version with basic countries');
                var basicCountries = [
                    'Netherlands', 'Belgium', 'Germany', 'France', 'United Kingdom', 
                    'Spain', 'Italy', 'Portugal', 'Austria', 'Switzerland', 'Denmark', 
                    'Sweden', 'Norway', 'Finland', 'Poland', 'Czech Republic', 'Hungary', 
                    'Romania', 'Bulgaria', 'Croatia', 'Slovenia', 'Slovakia', 'Estonia', 
                    'Latvia', 'Lithuania', 'Ireland', 'Luxembourg', 'Malta', 'Cyprus', 
                    'United States', 'Canada', 'Australia', 'New Zealand'
                ];
                
                // Populate country selects
                var countrySelects = ['#country-select', '#bulk-delete-country'];
                countrySelects.forEach(function(selector) {
                    var $select = $(selector);
                    if ($select.length > 0) {
                        // Clear existing options except first
                        $select.find('option:not(:first)').remove();
                        
                        // Add basic countries
                        basicCountries.forEach(function(country) {
                            $select.append('<option value="' + country + '">' + country + '</option>');
                        });
                    }
                });
            }
            
            // Bulk delete functionality
            function loadBulkDeleteOptions() {
                debugLog('loadBulkDeleteOptions', 'Starting to load bulk delete options');
                
                // Ensure ajaxurl is available
                var ajaxUrl = (typeof ajaxurl !== 'undefined') ? ajaxurl : '<?php echo admin_url('admin-ajax.php'); ?>';
                
                // Load countries
                $.ajax({
                    url: ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'get_bulk_delete_countries',
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        debugLog('loadBulkDeleteOptions', 'Countries AJAX call successful', response);
                        if (response.success) {
                            debugLog('loadBulkDeleteOptions', 'Countries response indicates success, updating UI');
                            $('#bulk-delete-country').html(response.data);
                        } else {
                            debugLog('loadBulkDeleteOptions', 'Countries response indicates failure', response);
                        }
                    },
                    error: function(xhr, status, error) {
                        debugLog('loadBulkDeleteOptions', 'Countries AJAX call failed', {status: status, error: error, responseText: xhr.responseText});
                        console.error('Error loading bulk delete countries:', error);
                    }
                });
                
                // Load provinces
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_bulk_delete_provinces',
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        debugLog('loadBulkDeleteOptions', 'Provinces AJAX call successful', response);
                        if (response.success) {
                            debugLog('loadBulkDeleteOptions', 'Provinces response indicates success, updating UI');
                            $('#bulk-delete-province').html(response.data);
                        } else {
                            debugLog('loadBulkDeleteOptions', 'Provinces response indicates failure', response);
                        }
                    },
                    error: function(xhr, status, error) {
                        debugLog('loadBulkDeleteOptions', 'Provinces AJAX call failed', {status: status, error: error, responseText: xhr.responseText});
                        console.error('Error loading bulk delete provinces:', error);
                    }
                });
            }
            
            // Update delete count when filters change
            $('#bulk-delete-country, #bulk-delete-province, input[name="delete_type"]').on('change', function() {
                updateDeleteCount();
            });
            
            function updateDeleteCount() {
                var country = $('#bulk-delete-country').val();
                var province = $('#bulk-delete-province').val();
                var deleteType = $('input[name="delete_type"]:checked').val();
                
                if (!country && !province) {
                    $('#bulk-delete-count').text('');
                    return;
                }
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_bulk_delete_count',
                        country: country,
                        province: province,
                        delete_type: deleteType,
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            var count = response.data.count;
                            if (count > 0) {
                                $('#bulk-delete-count').text('(' + count + ' locations will be deleted)');
                            } else {
                                $('#bulk-delete-count').text('(No locations match criteria)');
                            }
                        }
                    }
                });
            }
            
            // Saved Locations Bulk Delete Functions
            function loadSavedLocationsForBulkDelete() {
                // Load countries for saved locations bulk delete
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_bulk_delete_countries',
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#saved-locations-country').html(response.data);
                        }
                    }
                });
                
                // Load saved locations
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_saved_locations_for_bulk_delete',
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#saved-locations-select').html(response.data);
                        }
                    }
                });
            }
            
            // Country filter change for saved locations
            $('#saved-locations-country').on('change', function() {
                var country = $(this).val();
                
                // Load provinces for selected country
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_bulk_delete_provinces',
                        country: country,
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#saved-locations-province').html(response.data);
                        }
                    }
                });
                
                // Filter saved locations
                filterSavedLocations();
            });
            
            // Province filter change for saved locations
            $('#saved-locations-province').on('change', function() {
                filterSavedLocations();
            });
            
            function filterSavedLocations() {
                var country = $('#saved-locations-country').val();
                var province = $('#saved-locations-province').val();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_saved_locations_for_bulk_delete',
                        country: country,
                        province: province,
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#saved-locations-select').html(response.data);
                            updateSavedLocationsSelectedCount();
                        }
                    }
                });
            }
            
            // Select all visible saved locations
            $('#select-all-saved-locations').on('click', function() {
                $('#saved-locations-select option').prop('selected', true);
                updateSavedLocationsSelectedCount();
            });
            
            // Clear saved locations selection
            $('#clear-saved-locations-selection').on('click', function() {
                $('#saved-locations-select option').prop('selected', false);
                updateSavedLocationsSelectedCount();
            });
            
            // Update saved locations selection count
            function updateSavedLocationsSelectedCount() {
                var selectedCount = $('#saved-locations-select option:selected').length;
                $('#saved-locations-selected-count').text(selectedCount + ' locations selected');
            }
            
            // Saved locations selection change
            $('#saved-locations-select').on('change', function() {
                updateSavedLocationsSelectedCount();
            });
            
            // Bulk delete saved locations button
            $('#bulk-delete-saved-locations').on('click', function() {
                var selectedLocations = $('#saved-locations-select option:selected').map(function() {
                    return $(this).val();
                }).get();
                
                if (selectedLocations.length === 0) {
                    alert('Please select at least one location to delete.');
                    return;
                }
                
                if (confirm('Are you sure you want to delete ' + selectedLocations.length + ' selected locations? This action cannot be undone!')) {
                    var $button = $(this);
                    var originalText = $button.text();
                    
                    $button.prop('disabled', true).text('Deleting...');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'bulk_delete_saved_locations',
                            locations: selectedLocations,
                            nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                $('#saved-locations-bulk-result').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                                loadSavedLocations(); // Reload the locations list
                                loadSavedLocationsForBulkDelete(); // Reload the bulk delete options
                            } else {
                                $('#saved-locations-bulk-result').html('<div class="notice notice-error"><p>Error: ' + response.data + '</p></div>');
                            }
                        },
                        error: function() {
                            $('#saved-locations-bulk-result').html('<div class="notice notice-error"><p>Error occurred while deleting locations.</p></div>');
                        },
                        complete: function() {
                            $button.prop('disabled', false).text(originalText);
                        }
                    });
                }
            });
            
            // Bulk delete button
            $('#bulk-delete-btn').on('click', function() {
                var country = $('#bulk-delete-country').val();
                var province = $('#bulk-delete-province').val();
                var deleteType = $('input[name="delete_type"]:checked').val();
                
                if (!country && !province) {
                    alert('Please select a country or province/state to delete.');
                    return;
                }
                
                var confirmMessage = 'Are you sure you want to delete all locations';
                if (country) {
                    confirmMessage += ' from ' + country;
                }
                if (province) {
                    confirmMessage += ' in ' + province;
                }
                confirmMessage += '? This action cannot be undone!';
                
                if (confirm(confirmMessage)) {
                    var $button = $(this);
                    var originalText = $button.text();
                    
                    $button.prop('disabled', true).text('Deleting...');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'bulk_delete_locations',
                            country: country,
                            province: province,
                            delete_type: deleteType,
                            nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                $('#bulk-delete-result').html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                                $('#bulk-delete-count').text('');
                                $('#bulk-delete-country, #bulk-delete-province').val('');
                                loadSavedLocations(); // Reload the locations list
                                loadBulkDeleteOptions(); // Reload the dropdowns
                            } else {
                                $('#bulk-delete-result').html('<div class="notice notice-error"><p>' + response.data + '</p></div>');
                            }
                        },
                        error: function() {
                            $('#bulk-delete-result').html('<div class="notice notice-error"><p>Error occurred while deleting locations.</p></div>');
                        },
                        complete: function() {
                            $button.prop('disabled', false).text(originalText);
                        }
                    });
                }
            });
        });
        </script>
        <?php
    }
    
    public function template_maker_page() {
        // Ensure premium status is checked when page is loaded
        $this->check_premium_status();
        $this->premium_addon_notice();
        ?>
        <div class="wrap">
            <h1>Template Maker</h1>
            <p>Create custom templates for your lead pages.</p>
            
            <div class="notice notice-info" style="margin: 20px 0; padding: 15px;">
                <h4 style="margin-top: 0;">🎨 Advanced Template Features</h4>
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                    <div>
                        <h5>🎯 Spintax Syntax</h5>
                        <p>Create dynamic content with spintax variations:</p>
                        <ul>
                            <li><code>{Professional|Expert|Skilled|Qualified}</code> - Random adjectives</li>
                            <li><code>{plumbing|electrical|painting|gardening} services</code> - Service types</li>
                            <li><code>{fast|quick|rapid|immediate} response</code> - Speed variations</li>
                            <li><code>{24/7|around the clock|24 hours a day|anytime}</code> - Availability</li>
                            <li><code>{affordable|competitive|reasonable|fair} prices</code> - Pricing</li>
                            <li><code>{local|nearby|in your area|close to you}</code> - Location references</li>
                        </ul>
                        <p><strong>Example:</strong> "We provide {Professional|Expert|Skilled} {plumbing|electrical|painting} services with {fast|quick|rapid} response times."</p>
                    </div>
                    <div>
                        <h5>🔗 Dynamic Shortcodes</h5>
                        <p>Use these shortcodes for dynamic content:</p>
                        <ul>
                            <li><code>[service_name]</code> - Main service name</li>
                            <li><code>[location_name]</code> - City/town name</li>
                            <li><code>[province_name]</code> - Province/state</li>
                            <li><code>[phone_number]</code> - Contact phone</li>
                            <li><code>[contact_email]</code> - Contact email</li>
                            <li><code>[contact_form]</code> - Lead capture form</li>
                            <li><code>[business_name]</code> - Your business name</li>
                            <li><code>[website_url]</code> - Your website URL</li>
                        </ul>
                        <p><strong>Example:</strong> "Contact [business_name] for [service_name] in [location_name], [province_name] at [phone_number]."</p>
                    </div>
                </div>
                
                <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin-top: 15px;">
                    <h5 style="margin-top: 0;">💡 Pro Tips</h5>
                    <ul style="margin-bottom: 0;">
                        <li><strong>Combine spintax with shortcodes:</strong> "We offer {Professional|Expert} [service_name] services in [location_name]"</li>
                        <li><strong>Use multiple variations:</strong> Create 3-5 variations for each spintax to avoid repetition</li>
                        <li><strong>Test your templates:</strong> Preview generated content to ensure natural flow</li>
                        <li><strong>Keep it natural:</strong> Don't overuse spintax - aim for readability</li>
                    </ul>
                </div>
            </div>
            
            <form id="template-form">
                <?php wp_nonce_field('seo_lead_pages_nonce', 'nonce'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Template Name</th>
                        <td><input type="text" name="template_name" class="regular-text" required /></td>
                    </tr>
                    <tr>
                        <th scope="row">Template Content</th>
                        <td>
                            <textarea name="template_content" rows="10" cols="50" class="large-text" placeholder="Enter your template content here. Use shortcodes like [service_name], [location_name] and spintax like {Professional|Expert|Skilled} for dynamic content."></textarea>
                            <p class="description">Use shortcodes and spintax syntax to create dynamic, SEO-optimized content that varies for each location.</p>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <input type="submit" class="button-primary" value="Save Template" />
                </p>
            </form>
            
            <div id="template-result"></div>
            
            <!-- Template Overview Section -->
            <div style="margin-top: 40px;">
                <h2>📋 Saved Templates</h2>
                <p>Manage your saved templates below. You can edit or delete existing templates.</p>
                
                <div id="templates-overview">
                    <div style="text-align: center; padding: 20px;">
                        <span class="spinner is-active"></span>
                        <p>Loading templates...</p>
                    </div>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Load templates on page load
            loadTemplates();
            
            // Template form submission
            $('#template-form').submit(function(e) {
                e.preventDefault();
                
                var formData = new FormData(this);
                formData.append('action', 'save_template');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        if (response.success) {
                            $('#template-result').html('<div class="notice notice-success"><p>' + response.data + '</p></div>');
                            $('#template-form')[0].reset();
                            loadTemplates(); // Reload templates after saving
                        } else {
                            $('#template-result').html('<div class="notice notice-error"><p>' + response.data + '</p></div>');
                        }
                    },
                    error: function(xhr, status, error) {
                        $('#template-result').html('<div class="notice notice-error"><p>Error saving template: ' + error + '</p></div>');
                    }
                });
            });
            
            // Load templates function
            function loadTemplates() {
                console.log('Loading templates...');
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'get_templates_overview',
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        console.log('Templates response:', response);
                        if (response.success && response.data.templates) {
                            console.log('Templates found:', response.data.templates);
                            displayTemplates(response.data.templates);
                        } else {
                            console.log('No templates found or error in response');
                            $('#templates-overview').html('<div class="notice notice-info"><p>No templates found. Create your first template above!</p></div>');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('Error loading templates:', error);
                        $('#templates-overview').html('<div class="notice notice-error"><p>Error loading templates: ' + error + '</p></div>');
                    }
                });
            }
            
            // Display templates function
            function displayTemplates(templates) {
                if (templates.length === 0) {
                    $('#templates-overview').html('<div class="notice notice-info"><p>No templates found. Create your first template above!</p></div>');
                    return;
                }
                
                var html = '<div class="templates-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 20px; margin-top: 20px;">';
                
                templates.forEach(function(template) {
                    html += '<div class="template-card" style="border: 1px solid #ddd; border-radius: 8px; padding: 20px; background: #fff;">';
                    html += '<h4 style="margin-top: 0; color: #0073aa;">' + template.name + '</h4>';
                    html += '<div style="max-height: 100px; overflow-y: auto; margin: 10px 0; padding: 10px; background: #f8f9fa; border-radius: 4px; font-size: 12px;">';
                    html += '<pre style="margin: 0; white-space: pre-wrap;">' + template.content.substring(0, 200) + (template.content.length > 200 ? '...' : '') + '</pre>';
                    html += '</div>';
                    html += '<div style="display: flex; gap: 10px; margin-top: 15px;">';
                    html += '<button class="button button-secondary edit-template" data-template-name="' + template.name + '" data-template-content="' + template.content.replace(/"/g, '&quot;') + '">Edit</button>';
                    html += '<button class="button button-link-delete delete-template" data-template-name="' + template.name + '">Delete</button>';
                    html += '</div>';
                    html += '</div>';
                });
                
                html += '</div>';
                $('#templates-overview').html(html);
                
                // Bind edit and delete events
                $('.edit-template').click(function() {
                    var name = $(this).data('template-name');
                    var content = $(this).data('template-content');
                    $('input[name="template_name"]').val(name);
                    $('textarea[name="template_content"]').val(content);
                    $('html, body').animate({scrollTop: $('#template-form').offset().top}, 500);
                });
                
                $('.delete-template').click(function() {
                    if (confirm('Are you sure you want to delete this template?')) {
                        var name = $(this).data('template-name');
                        deleteTemplate(name);
                    }
                });
            }
            
            // Delete template function
            function deleteTemplate(templateName) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'delete_template',
                        template_name: templateName,
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#template-result').html('<div class="notice notice-success"><p>' + response.data + '</p></div>');
                            loadTemplates(); // Reload templates after deletion
                        } else {
                            $('#template-result').html('<div class="notice notice-error"><p>' + response.data + '</p></div>');
                        }
                    },
                    error: function() {
                        $('#template-result').html('<div class="notice notice-error"><p>Error deleting template.</p></div>');
                    }
                });
            }
        });
        </script>
        <?php
    }
    
    public function analytics_page() {
        // Ensure premium status is checked when page is loaded
        $this->check_premium_status();
        $this->premium_addon_notice();
        
        // Get leads data
        global $wpdb;
        $leads = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}seo_lead_pages_leads ORDER BY created_at DESC LIMIT 50");
        
        // Get page views data
        $page_views = $wpdb->get_results("
            SELECT page_id, COUNT(*) as view_count 
            FROM {$wpdb->prefix}seo_lead_pages_page_views 
            GROUP BY page_id 
            ORDER BY view_count DESC 
            LIMIT 10
        ");
        
        // Get lead status counts
        $status_counts = $wpdb->get_results("
            SELECT status, COUNT(*) as count 
            FROM {$wpdb->prefix}seo_lead_pages_leads 
            GROUP BY status
        ");
        
        ?>
        <?php if ($this->get_premium_active()): ?>
        <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
        <?php endif; ?>
        
        <div class="wrap">
            <h1>Analytics Dashboard</h1>
            <p>View analytics and manage leads for your lead pages.</p>
            
            <!-- Statistics Overview -->
            <div class="notice notice-info" style="margin: 20px 0; padding: 15px;">
                <h3 style="margin-top: 0;">📊 Basic Statistics (Free Version)</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin: 15px 0;">
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; text-align: center;">
                        <h4 style="margin: 0; color: #0073aa;">Total Leads</h4>
                        <p style="font-size: 24px; font-weight: bold; margin: 5px 0;"><?php echo count($leads); ?></p>
                    </div>
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; text-align: center;">
                        <h4 style="margin: 0; color: #0073aa;">Total Page Views</h4>
                        <p style="font-size: 24px; font-weight: bold; margin: 5px 0;"><?php echo array_sum(array_column($page_views, 'view_count')); ?></p>
                    </div>
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; text-align: center;">
                        <h4 style="margin: 0; color: #0073aa;">Active Pages</h4>
                        <p style="font-size: 24px; font-weight: bold; margin: 5px 0;"><?php echo count($page_views); ?></p>
                    </div>
                </div>
            </div>
            
            <!-- Top 10 Pages -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>🏆 Top 10 Most Visited Landing Pages</h3>
                <?php if (!empty($page_views)): ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th>Rank</th>
                                <th>Page URL</th>
                                <th>Views</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($page_views as $index => $page): ?>
                                <?php 
                                $page_url = get_permalink($page->page_id);
                                $page_title = get_the_title($page->page_id);
                                ?>
                                <tr>
                                    <td><?php echo $index + 1; ?></td>
                                    <td>
                                        <a href="<?php echo esc_url($page_url); ?>" target="_blank">
                                            <?php echo esc_html($page_title ? $page_title : 'Page ID: ' . $page->page_id); ?>
                                        </a>
                                    </td>
                                    <td><strong><?php echo $page->view_count; ?></strong></td>
                                    <td>
                                        <a href="<?php echo esc_url($page_url); ?>" target="_blank" class="button button-small">View Page</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p>No page views recorded yet. Start generating pages to see statistics!</p>
                <?php endif; ?>
            </div>
            
            <!-- Lead Management -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>📋 Lead Management</h3>
                
                <!-- Lead Status Overview -->
                <?php if (!empty($status_counts)): ?>
                    <div style="display: flex; gap: 15px; margin: 15px 0; flex-wrap: wrap;">
                        <?php foreach ($status_counts as $status): ?>
                            <div style="background: #f8f9fa; padding: 10px 15px; border-radius: 6px; text-align: center;">
                                <strong><?php echo ucfirst($status->status ?: 'new'); ?></strong><br>
                                <span style="font-size: 18px; color: #0073aa;"><?php echo $status->count; ?></span>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
                
                <!-- Leads Table -->
                <?php if (!empty($leads)): ?>
                    <form id="leads-management-form">
                        <?php wp_nonce_field('seo_lead_pages_nonce', 'nonce'); ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th style="width: 20px;"><input type="checkbox" id="select-all-leads"></th>
                                    <th>Name</th>
                                    <th>Email</th>
                                    <th>Phone</th>
                                    <th>Location</th>
                                    <th>Service</th>
                                    <th>Status</th>
                                    <th>Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($leads as $lead): ?>
                                    <tr>
                                        <td><input type="checkbox" name="lead_ids[]" value="<?php echo $lead->id; ?>"></td>
                                        <td><?php echo esc_html($lead->name); ?></td>
                                        <td><?php echo esc_html($lead->email); ?></td>
                                        <td><?php echo esc_html($lead->phone); ?></td>
                                        <td><?php echo esc_html($lead->location); ?></td>
                                        <td><?php echo esc_html($lead->service); ?></td>
                                        <td>
                                            <select name="status[<?php echo $lead->id; ?>]" class="lead-status" data-lead-id="<?php echo $lead->id; ?>">
                                                <option value="new" <?php selected($lead->status, 'new'); ?>>New</option>
                                                <option value="open" <?php selected($lead->status, 'open'); ?>>Open</option>
                                                <option value="in_progress" <?php selected($lead->status, 'in_progress'); ?>>In Progress</option>
                                                <option value="rejected" <?php selected($lead->status, 'rejected'); ?>>Rejected</option>
                                                <option value="completed" <?php selected($lead->status, 'completed'); ?>>Completed</option>
                                            </select>
                                        </td>
                                        <td><?php echo date('Y-m-d H:i', strtotime($lead->created_at)); ?></td>
                                        <td>
                                            <button type="button" class="button button-small delete-lead" data-lead-id="<?php echo $lead->id; ?>">Delete</button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        
                        <div style="margin-top: 15px;">
                            <button type="button" id="bulk-delete-leads" class="button button-secondary">Delete Selected</button>
                            <button type="button" id="export-leads-csv" class="button button-secondary">Export CSV</button>
                        </div>
                    </form>
                <?php else: ?>
                    <p>No leads collected yet. Start generating pages to collect leads!</p>
                <?php endif; ?>
            </div>
            
            <?php if ($this->get_premium_active()): ?>
            <!-- Advanced Analytics (Premium) -->
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>📊 Advanced Analytics</h3>
                <p>Premium analytics with detailed insights and conversion tracking.</p>
                
                <div style="margin-bottom:20px;">
                    <label for="analytics-period">Period:</label>
                    <select id="analytics-period" style="margin-left:10px;">
                        <option value="7">Last 7 days</option>
                        <option value="30" selected>Last 30 days</option>
                        <option value="90">Last 90 days</option>
                        <option value="365">Last year</option>
                    </select>
                    <button type="button" id="refresh-analytics" class="button button-primary" style="margin-left:10px;">
                        Refresh
                    </button>
                </div>
                
                <!-- Premium Stats Overview -->
                <div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(200px,1fr));gap:20px;margin-bottom:30px;">
                    <div style="background:#fff;padding:20px;border-radius:12px;box-shadow:0 4px 20px rgba(0,0,0,0.1);text-align:center;">
                        <div id="pages-generated" style="font-size:32px;font-weight:bold;color:#007cba;">-</div>
                        <div style="color:#666;margin-top:5px;">Pages Generated</div>
                    </div>
                    <div style="background:#fff;padding:20px;border-radius:12px;box-shadow:0 4px 20px rgba(0,0,0,0.1);text-align:center;">
                        <div id="leads-received" style="font-size:32px;font-weight:bold;color:#28a745;">-</div>
                        <div style="color:#666;margin-top:5px;">Leads Received</div>
                    </div>
                    <div style="background:#fff;padding:20px;border-radius:12px;box-shadow:0 4px 20px rgba(0,0,0,0.1);text-align:center;">
                        <div id="page-views" style="font-size:32px;font-weight:bold;color:#ffc107;">-</div>
                        <div style="color:#666;margin-top:5px;">Page Views</div>
                    </div>
                    <div style="background:#fff;padding:20px;border-radius:12px;box-shadow:0 4px 20px rgba(0,0,0,0.1);text-align:center;">
                        <div id="conversion-rate" style="font-size:32px;font-weight:bold;color:#dc3545;">-</div>
                        <div style="color:#666;margin-top:5px;">Conversion Rate %</div>
                    </div>
                </div>
                
                <!-- Premium Charts -->
                <div style="display:grid;grid-template-columns:1fr 1fr;gap:20px;margin-bottom:30px;">
                    <div style="background:#fff;padding:25px;border-radius:12px;box-shadow:0 4px 20px rgba(0,0,0,0.1);">
                        <h3 style="margin-top:0;">Monthly Statistics</h3>
                        <canvas id="monthly-chart" width="400" height="200"></canvas>
                    </div>
                    <div style="background:#fff;padding:25px;border-radius:12px;box-shadow:0 4px 20px rgba(0,0,0,0.1);">
                        <h3 style="margin-top:0;">Top Locations</h3>
                        <canvas id="locations-chart" width="400" height="200"></canvas>
                    </div>
                </div>
                
                <!-- Top Pages Table -->
                <div style="background:#fff;padding:25px;border-radius:12px;box-shadow:0 4px 20px rgba(0,0,0,0.1);margin-bottom:30px;">
                    <h3 style="margin-top:0;">Top Performing Pages</h3>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th>Page Title</th>
                                <th>Views</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="top-pages-table">
                            <tr>
                                <td colspan="3">Loading...</td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            <?php if ($this->get_premium_active()): ?>
            // Premium Analytics functionality
            let monthlyChart = null;
            let locationsChart = null;
            
            // Load analytics on page load
            loadPremiumAnalytics();
            
            // Refresh button
            $('#refresh-analytics').on('click', function() {
                loadPremiumAnalytics();
            });
            
            // Period change
            $('#analytics-period').on('change', function() {
                loadPremiumAnalytics();
            });
            
            function loadPremiumAnalytics() {
                var period = $('#analytics-period').val();
                
                $.post(ajaxurl, {
                    action: 'geo_local_seo_premium_get_analytics',
                    nonce: '<?php echo wp_create_nonce('geo_local_seo_premium_nonce'); ?>',
                    period: period
                }, function(response) {
                    if (response.success) {
                        updatePremiumStats(response.data);
                        updatePremiumCharts(response.data);
                        updatePremiumTopPagesTable(response.data.top_pages);
                    } else {
                        console.error('Failed to load premium analytics:', response.data.message);
                    }
                });
            }
            
            function updatePremiumStats(data) {
                $('#pages-generated').text(data.pages_generated || 0);
                $('#leads-received').text(data.leads_received || 0);
                $('#page-views').text(data.page_views || 0);
                $('#conversion-rate').text(data.conversion_rate || 0);
            }
            
            function updatePremiumCharts(data) {
                // Monthly chart
                if (monthlyChart) {
                    monthlyChart.destroy();
                }
                
                var monthlyCtx = document.getElementById('monthly-chart').getContext('2d');
                monthlyChart = new Chart(monthlyCtx, {
                    type: 'line',
                    data: {
                        labels: data.monthly_stats.map(item => item.month),
                        datasets: [{
                            label: 'Pages Generated',
                            data: data.monthly_stats.map(item => item.pages_generated),
                            borderColor: '#007cba',
                            backgroundColor: 'rgba(0, 124, 186, 0.1)',
                            tension: 0.1
                        }, {
                            label: 'Leads Received',
                            data: data.monthly_stats.map(item => item.leads_received),
                            borderColor: '#28a745',
                            backgroundColor: 'rgba(40, 167, 69, 0.1)',
                            tension: 0.1
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
                
                // Locations chart
                if (locationsChart) {
                    locationsChart.destroy();
                }
                
                var locationsCtx = document.getElementById('locations-chart').getContext('2d');
                locationsChart = new Chart(locationsCtx, {
                    type: 'doughnut',
                    data: {
                        labels: data.top_locations.map(item => item.location),
                        datasets: [{
                            data: data.top_locations.map(item => item.page_count),
                            backgroundColor: [
                                '#007cba',
                                '#28a745',
                                '#ffc107',
                                '#dc3545',
                                '#6f42c1',
                                '#20c997',
                                '#fd7e14',
                                '#6c757d'
                            ]
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'bottom'
                            }
                        }
                    }
                });
            }
            
            function updatePremiumTopPagesTable(pages) {
                var html = '';
                if (pages && pages.length > 0) {
                    pages.forEach(function(page) {
                        html += '<tr>';
                        html += '<td>' + page.post_title + '</td>';
                        html += '<td>' + (page.total_views || 0) + '</td>';
                        html += '<td><a href="<?php echo admin_url('post.php?post='); ?>' + page.ID + '&action=edit" class="button button-small">Edit</a></td>';
                        html += '</tr>';
                    });
                } else {
                    html = '<tr><td colspan="3">No data available</td></tr>';
                }
                $('#top-pages-table').html(html);
            }
            <?php endif; ?>
            
            // Select all checkbox
            $('#select-all-leads').change(function() {
                $('input[name="lead_ids[]"]').prop('checked', this.checked);
            });
            
            // Update lead status
            $('.lead-status').change(function() {
                var leadId = $(this).data('lead-id');
                var status = $(this).val();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'update_lead_status',
                        lead_id: leadId,
                        status: status,
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            // Show success message
                            $('<div class="notice notice-success"><p>Lead status updated successfully!</p></div>')
                                .insertAfter('.wrap h1')
                                .delay(3000)
                                .fadeOut();
                        }
                    }
                });
            });
            
            // Delete single lead
            $('.delete-lead').click(function() {
                if (confirm('Are you sure you want to delete this lead?')) {
                    var leadId = $(this).data('lead-id');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'delete_lead',
                            lead_id: leadId,
                            nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                location.reload();
                            }
                        }
                    });
                }
            });
            
            // Bulk delete leads
            $('#bulk-delete-leads').click(function() {
                var selectedLeads = $('input[name="lead_ids[]"]:checked').map(function() {
                    return this.value;
                }).get();
                
                if (selectedLeads.length === 0) {
                    alert('Please select leads to delete.');
                    return;
                }
                
                if (confirm('Are you sure you want to delete ' + selectedLeads.length + ' selected leads?')) {
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'bulk_delete_leads',
                            lead_ids: selectedLeads,
                            nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                location.reload();
                            }
                        }
                    });
                }
            });
            
            // Export CSV
            $('#export-leads-csv').click(function() {
                window.location.href = ajaxurl + '?action=export_leads_csv&nonce=<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>';
            });
            
            <?php if ($this->get_premium_active()): ?>
            // Premium Analytics functionality
            let monthlyChart = null;
            let locationsChart = null;
            
            // Load analytics on page load
            loadAnalytics();
            
            // Refresh button
            $('#refresh-analytics').on('click', function() {
                loadAnalytics();
            });
            
            // Period change
            $('#analytics-period').on('change', function() {
                loadAnalytics();
            });
            
            function loadAnalytics() {
                var period = $('#analytics-period').val();
                
                $.post(ajaxurl, {
                    action: 'geo_local_seo_premium_get_analytics',
                    nonce: '<?php echo wp_create_nonce('geo_local_seo_premium_nonce'); ?>',
                    period: period
                }, function(response) {
                    if (response.success) {
                        updateStats(response.data);
                        updateCharts(response.data);
                        updateTopPagesTable(response.data.top_pages);
                    } else {
                        console.error('Failed to load analytics:', response.data.message);
                    }
                });
            }
            
            function updateStats(data) {
                $('#pages-generated').text(data.pages_generated || 0);
                $('#leads-received').text(data.leads_received || 0);
                $('#page-views').text(data.page_views || 0);
                $('#conversion-rate').text(data.conversion_rate || 0);
            }
            
            function updateCharts(data) {
                // Monthly chart
                if (monthlyChart) {
                    monthlyChart.destroy();
                }
                
                var monthlyCtx = document.getElementById('monthly-chart').getContext('2d');
                monthlyChart = new Chart(monthlyCtx, {
                    type: 'line',
                    data: {
                        labels: data.monthly_stats.map(item => item.month),
                        datasets: [{
                            label: 'Pages Generated',
                            data: data.monthly_stats.map(item => item.pages_generated),
                            borderColor: '#007cba',
                            backgroundColor: 'rgba(0, 124, 186, 0.1)',
                            tension: 0.1
                        }, {
                            label: 'Leads Received',
                            data: data.monthly_stats.map(item => item.leads_received),
                            borderColor: '#28a745',
                            backgroundColor: 'rgba(40, 167, 69, 0.1)',
                            tension: 0.1
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
                
                // Locations chart
                if (locationsChart) {
                    locationsChart.destroy();
                }
                
                var locationsCtx = document.getElementById('locations-chart').getContext('2d');
                locationsChart = new Chart(locationsCtx, {
                    type: 'doughnut',
                    data: {
                        labels: data.top_locations.map(item => item.location),
                        datasets: [{
                            data: data.top_locations.map(item => item.page_count),
                            backgroundColor: [
                                '#007cba',
                                '#28a745',
                                '#ffc107',
                                '#dc3545',
                                '#6f42c1',
                                '#20c997',
                                '#fd7e14',
                                '#6c757d'
                            ]
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'bottom'
                            }
                        }
                    }
                });
            }
            
            function updateTopPagesTable(pages) {
                var html = '';
                if (pages && pages.length > 0) {
                    pages.forEach(function(page) {
                        html += '<tr>';
                        html += '<td>' + page.post_title + '</td>';
                        html += '<td>' + (page.total_views || 0) + '</td>';
                        html += '<td><a href="' + '<?php echo admin_url('post.php?post='); ?>' + page.ID + '&action=edit" class="button button-small">Edit</a></td>';
                        html += '</tr>';
                    });
                } else {
                    html = '<tr><td colspan="3">No data available</td></tr>';
                }
                $('#top-pages-table').html(html);
            }
            <?php endif; ?>
            } catch (error) {
                console.error('[GEO SEO DEBUG] JavaScript error in Locations Manager:', error);
                console.error('[GEO SEO DEBUG] Error stack:', error.stack);
            }
        });
        </script>
        <?php
    }
    
    public function premium_info_page() {
        ?>
        <div class="wrap">
            <h1>🚀 Premium Addon - GEO Local SEO Lead Pages Generator</h1>
            
            <div class="notice notice-info" style="border-left-color: #0073aa; padding: 20px;">
                <h3 style="margin-top: 0;">✨ Upgrade to Premium - Unlock Unlimited Potential</h3>
                <p style="font-size: 16px; margin-bottom: 20px;">Transform your local SEO strategy with our comprehensive premium addon. Generate unlimited landing pages and dominate local search results.</p>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin: 20px 0;">
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                        <h4 style="color: #0073aa; margin-top: 0;">📊 Advanced Analytics Dashboard</h4>
                        <ul>
                            <li>Real-time conversion tracking</li>
                            <li>Lead generation statistics</li>
                            <li>Page views and performance metrics</li>
                            <li>Monthly statistics with charts</li>
                            <li>Top performing pages analysis</li>
                            <li>Conversion rate calculations</li>
                        </ul>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                        <h4 style="color: #0073aa; margin-top: 0;">⚡ Bulk Operations</h4>
                        <ul>
                            <li>Generate hundreds of pages instantly</li>
                            <li>Bulk SEO optimization for existing pages</li>
                            <li>Mass page generation with templates</li>
                            <li>Existing pages management</li>
                            <li>Batch processing with safety limits</li>
                            <li>Template and location selection</li>
                        </ul>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                        <h4 style="color: #0073aa; margin-top: 0;">🧪 A/B Testing</h4>
                        <ul>
                            <li>Create A/B tests for landing pages</li>
                            <li>Test different templates and content</li>
                            <li>Track conversion rates per variant</li>
                            <li>Statistical significance analysis</li>
                            <li>Automatic test management</li>
                            <li>Results reporting and insights</li>
                        </ul>
                    </div>
                    
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                        <h4 style="color: #0073aa; margin-top: 0;">🌍 Country Import System</h4>
                        <ul>
                            <li>Import thousands of locations instantly</li>
                            <li>Complete country databases</li>
                            <li>Cities with 20,000+ inhabitants</li>
                            <li>All provinces and states included</li>
                            <li>One-click import per country</li>
                            <li>Perfect for franchise businesses</li>
                        </ul>
                    </div>
                </div>
                
                <div style="background: #f0f8ff; padding: 20px; border-radius: 8px; margin: 20px 0;">
                    <h4 style="color: #0073aa; margin-top: 0;">🔗 Seamless Integration</h4>
                    <p><strong>Premium features integrate directly into your existing workflow:</strong></p>
                    <ul style="columns: 2; column-gap: 30px;">
                        <li>✅ <strong>Locations Manager</strong> - Country import appears automatically</li>
                        <li>✅ <strong>Page Generator</strong> - Bulk operations integrated seamlessly</li>
                        <li>✅ <strong>Analytics Dashboard</strong> - Advanced analytics enhance existing stats</li>
                        <li>✅ <strong>Template System</strong> - Works with your custom templates</li>
                        <li>✅ <strong>A/B Testing</strong> - Dedicated testing management</li>
                        <li>✅ <strong>License Management</strong> - Simple activation system</li>
                    </ul>
                    <p style="margin-bottom: 0;"><strong>No separate interfaces needed</strong> - everything works together as one unified system!</p>
                </div>
                
                <div style="background: #e7f3ff; padding: 20px; border-radius: 8px; margin: 20px 0;">
                    <h4 style="color: #0073aa; margin-top: 0;">🌍 Mass Location Import</h4>
                    <p><strong>Import entire countries at once!</strong> Load all locations from:</p>
                    <ul style="columns: 3; column-gap: 20px;">
                        <li>🇳🇱 Netherlands</li>
                        <li>🇧🇪 Belgium</li>
                        <li>🇩🇪 Germany</li>
                        <li>🇫🇷 France</li>
                        <li>🇬🇧 United Kingdom</li>
                        <li>🇪🇸 Spain</li>
                        <li>🇮🇹 Italy</li>
                        <li>🇺🇸 United States (East, West, Central)</li>
                        <li>🇨🇦 Canada</li>
                        <li>🇦🇺 Australia</li>
                        <li>🇵🇹 Portugal</li>
                        <li>🇦🇹 Austria</li>
                        <li>🇨🇭 Switzerland</li>
                        <li>🇵🇱 Poland</li>
                        <li>🇮🇪 Ireland</li>
                        <li>🇮🇱 Israel</li>
                        <li>🇲🇽 Mexico</li>
                        <li>🇧🇷 Brazil</li>
                        <li>🇲🇦 Morocco</li>
                        <li>🇹🇷 Turkey</li>
                        <li>🇮🇳 India</li>
                        <li>🇨🇳 China</li>
                        <li>🇯🇵 Japan</li>
                        <li>🇰🇷 South Korea</li>
                        <li>🇮🇩 Indonesia</li>
                    </ul>
                    <p style="margin-bottom: 0;"><strong>Perfect for:</strong> Franchise businesses, multi-location organizations, SEO agencies, and marketing agencies.</p>
                </div>
                
                <div style="background: #fff3cd; padding: 20px; border-radius: 8px; margin: 20px 0;">
                    <h4 style="color: #856404; margin-top: 0;">💰 Proven Results & ROI</h4>
                    <p style="margin-bottom: 10px;"><strong>Typical results: 200-500% ROI within 3-6 months</strong> through increased local rankings and lead generation.</p>
                    <p style="margin-bottom: 0;"><strong>⚡ What to Expect:</strong></p>
                    <ul style="margin-bottom: 0;">
                        <li>Within 24 hours: Landing pages live and indexed</li>
                        <li>Within 1-2 weeks: Improved local search visibility</li>
                        <li>Within 1 month: Measurable lead increase</li>
                        <li>Within 3 months: Significant ranking improvements</li>
                        <li>Within 6 months: Full ROI realization</li>
                    </ul>
                </div>
                
                <div style="background: #d1ecf1; padding: 20px; border-radius: 8px; margin: 20px 0;">
                    <h4 style="color: #0c5460; margin-top: 0;">🔒 Secure & Reliable</h4>
                    <ul style="margin-bottom: 0;">
                        <li>WordPress compatible</li>
                        <li>GDPR compliant</li>
                        <li>Regular updates</li>
                        <li>24/7 technical support</li>
                        <li>Money-back guarantee</li>
                    </ul>
                </div>
                
                <div style="text-align: center; margin: 30px 0;">
                    <a href="https://stormseo.nl/product/geo-local-seo-lead-page-generator-year-license/" target="_blank" class="button button-primary button-hero" style="font-size: 18px; padding: 15px 30px; background: #0073aa; border-color: #0073aa;">
                        🚀 Get Premium License - €49/year
                    </a>
                    <p style="margin-top: 10px; color: #666;">Start generating more leads and higher rankings for your local services today!</p>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function about_plugin_page() {
        ?>
        <div class="wrap">
            <h1>About GEO Local SEO Lead Pages Generator</h1>
            
            <div style="display: grid; grid-template-columns: 2fr 1fr; gap: 30px; margin: 20px 0;">
                <!-- Main Content -->
                <div>
                    <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px; margin-bottom: 20px;">
                        <h2 style="margin-top: 0; color: #0073aa;">🎯 What is this Plugin?</h2>
                        <p style="font-size: 16px; line-height: 1.6;">The GEO Local SEO Lead Pages Generator is a powerful WordPress plugin designed to automatically create SEO-optimized landing pages for local businesses. It helps you dominate local search results by generating location-specific pages that are perfectly tailored for Google's local search algorithm.</p>
                        
                        <h3>🚀 Key Features:</h3>
                        <ul style="font-size: 15px; line-height: 1.8;">
                            <li><strong>Automatic Page Generation:</strong> Create hundreds of location-based landing pages in minutes</li>
                            <li><strong>SEO Optimization:</strong> Built-in SEO best practices for local search rankings</li>
                            <li><strong>Template System:</strong> Customizable templates with spintax support for content variation</li>
                            <li><strong>Lead Management:</strong> Track and manage leads from your landing pages</li>
                            <li><strong>Analytics Dashboard:</strong> Monitor performance with detailed statistics</li>
                            <li><strong>Bulk Operations:</strong> Import locations and generate pages in bulk</li>
                        </ul>
                    </div>
                    
                    <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px; margin-bottom: 20px;">
                        <h2 style="margin-top: 0; color: #0073aa;">⚙️ How It Works</h2>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                            <div>
                                <h4>1. 📝 Page Generation</h4>
                                <p>Enter your business details and service information. The plugin automatically generates SEO-optimized content for each location.</p>
                                
                                <h4>2. 🎨 Template Customization</h4>
                                <p>Use our template maker to create custom layouts with spintax syntax for content variation and shortcodes for dynamic content.</p>
                            </div>
                            <div>
                                <h4>3. 📊 Lead Collection</h4>
                                <p>Visitors can contact you through integrated contact forms, and all leads are collected in your analytics dashboard.</p>
                                
                                <h4>4. 📈 Performance Tracking</h4>
                                <p>Monitor which pages perform best, track lead conversions, and optimize your local SEO strategy.</p>
                            </div>
                        </div>
                    </div>
                    
                    <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px;">
                        <h2 style="margin-top: 0; color: #0073aa;">💡 Perfect For</h2>
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                            <div style="background: #f8f9fa; padding: 15px; border-radius: 6px; text-align: center;">
                                <h4 style="margin: 0;">🏠 Local Services</h4>
                                <p style="margin: 5px 0 0;">Plumbers, electricians, painters, gardeners</p>
                            </div>
                            <div style="background: #f8f9fa; padding: 15px; border-radius: 6px; text-align: center;">
                                <h4 style="margin: 0;">🏢 Franchises</h4>
                                <p style="margin: 5px 0 0;">Multi-location businesses</p>
                            </div>
                            <div style="background: #f8f9fa; padding: 15px; border-radius: 6px; text-align: center;">
                                <h4 style="margin: 0;">🎯 SEO Agencies</h4>
                                <p style="margin: 5px 0 0;">Client local SEO campaigns</p>
                            </div>
                            <div style="background: #f8f9fa; padding: 15px; border-radius: 6px; text-align: center;">
                                <h4 style="margin: 0;">📱 Marketing Agencies</h4>
                                <p style="margin: 5px 0 0;">Lead generation campaigns</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Sidebar -->
                <div>
                    <!-- Author Info -->
                    <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px; margin-bottom: 20px;">
                        <h3 style="margin-top: 0; color: #0073aa;">👨‍💻 About the Author</h3>
                        <div style="text-align: center;">
                            <div style="background: #0073aa; color: white; width: 80px; height: 80px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 15px; font-size: 32px; font-weight: bold;">
                                TS
                            </div>
                            <h4 style="margin: 0;">Thomas F.L. Storm</h4>
                            <p style="margin: 5px 0; color: #666;">SEO Specialist</p>
                            <a href="https://nl.linkedin.com/in/interim-seo-specialist" target="_blank" class="button button-primary" style="margin-top: 10px;">
                                📱 Connect on LinkedIn
                            </a>
                        </div>
                        <p style="font-size: 14px; line-height: 1.6; margin-top: 15px;">Experienced SEO specialist with expertise in local search optimization, technical SEO, and WordPress development. Passionate about helping businesses improve their online visibility and generate more qualified leads.</p>
                    </div>
                    
                    <!-- Plugin Info -->
                    <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px; margin-bottom: 20px;">
                        <h3 style="margin-top: 0; color: #0073aa;">📋 Plugin Information</h3>
                        <table style="width: 100%;">
                            <tr>
                                <td><strong>Version:</strong></td>
                                <td>1.0.0</td>
                            </tr>
                            <tr>
                                <td><strong>WordPress:</strong></td>
                                <td>5.0+</td>
                            </tr>
                            <tr>
                                <td><strong>PHP:</strong></td>
                                <td>7.4+</td>
                            </tr>
                            <tr>
                                <td><strong>License:</strong></td>
                                <td>GPL v2+</td>
                            </tr>
                            <tr>
                                <td><strong>Last Updated:</strong></td>
                                <td><?php echo date('Y-m-d'); ?></td>
                            </tr>
                        </table>
                    </div>
                    
                    <!-- Support -->
                    <div style="background: white; padding: 25px; border: 1px solid #ddd; border-radius: 8px;">
                        <h3 style="margin-top: 0; color: #0073aa;">🆘 Support & Resources</h3>
                        <ul style="margin: 0;">
                            <li><a href="https://stormseo.nl" target="_blank">🌐 StormSEO Website</a></li>
                            <li><a href="https://stormseo.nl/product/geo-local-seo-lead-page-generator-year-license/" target="_blank">🚀 Premium Addon</a></li>
                            <li><a href="https://nl.linkedin.com/in/interim-seo-specialist" target="_blank">📱 LinkedIn Profile</a></li>
                            <li><a href="mailto:info@stormseo.nl">📧 Email Support</a></li>
                        </ul>
                        
                        <div style="background: #e7f3ff; padding: 15px; border-radius: 6px; margin-top: 15px;">
                            <h4 style="margin: 0 0 10px;">💡 Need Help?</h4>
                            <p style="margin: 0; font-size: 14px;">For technical support, feature requests, or custom development, feel free to reach out!</p>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Footer -->
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; text-align: center; margin-top: 30px;">
                <p style="margin: 0; color: #666;">
                    <strong>GEO Local SEO Lead Pages Generator</strong> - Helping businesses dominate local search results since 2025
                </p>
                <p style="margin: 5px 0 0; font-size: 14px; color: #999;">
                    Developed by <a href="https://nl.linkedin.com/in/interim-seo-specialist" target="_blank">Thomas F.L. Storm</a> | 
                    <a href="https://stormseo.nl" target="_blank">StormSEO</a>
                </p>
            </div>
        </div>
        <?php
    }
    
    // AJAX Handlers
    public function ajax_generate_lead_page() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $business_name = sanitize_text_field($_POST['business_name']);
        $location = sanitize_text_field($_POST['location']);
        $service = sanitize_text_field($_POST['service']);
        $phone = sanitize_text_field($_POST['phone']);
        $email = sanitize_email($_POST['email']);
        $website = esc_url_raw($_POST['website']);
        
        // Generate page content
        $title = "{$service} in {$location} - {$business_name}";
        $content = $this->generate_page_content($business_name, $location, $service, $phone, $email, $website);
        
        // Create post
        $post_id = wp_insert_post(array(
            'post_title' => $title,
            'post_content' => $content,
            'post_status' => 'publish',
            'post_type' => 'page'
        ));
        
        if ($post_id) {
            wp_send_json_success(array(
                'message' => 'Lead page generated successfully!',
                'post_id' => $post_id,
                'edit_url' => get_edit_post_link($post_id)
            ));
        } else {
            wp_send_json_error('Failed to generate lead page');
        }
    }
    
    public function ajax_cleanup_test_data() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        global $wpdb;
        
        // Delete test leads
        $wpdb->delete($wpdb->prefix . 'seo_lead_pages_leads', array('is_test' => 1));
        
        // Delete test page views
        $wpdb->delete($wpdb->prefix . 'seo_lead_pages_page_views', array('is_test' => 1));
        
        wp_send_json_success('Test data cleaned up successfully');
    }
    
    public function ajax_generate_sitemap() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $sitemap_content = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $sitemap_content .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        
        // Get all published pages
        $pages = get_posts(array(
            'post_type' => 'page',
            'post_status' => 'publish',
            'numberposts' => -1
        ));
        
        foreach ($pages as $page) {
            $sitemap_content .= '<url>' . "\n";
            $sitemap_content .= '<loc>' . get_permalink($page->ID) . '</loc>' . "\n";
            $sitemap_content .= '<lastmod>' . date('c', strtotime($page->post_modified)) . '</lastmod>' . "\n";
            $sitemap_content .= '<changefreq>weekly</changefreq>' . "\n";
            $sitemap_content .= '<priority>0.8</priority>' . "\n";
            $sitemap_content .= '</url>' . "\n";
        }
        
        $sitemap_content .= '</urlset>';
        
        // Save sitemap
        $upload_dir = wp_upload_dir();
        $sitemap_path = $upload_dir['basedir'] . '/sitemap.xml';
        
        if (file_put_contents($sitemap_path, $sitemap_content)) {
            wp_send_json_success('Sitemap generated successfully');
        } else {
            wp_send_json_error('Failed to generate sitemap');
        }
    }
    
    public function ajax_check_seo_issues() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $issues = array();
        
        // Check for pages without meta descriptions
        $pages = get_posts(array(
            'post_type' => 'page',
            'post_status' => 'publish',
            'numberposts' => -1
        ));
        
        foreach ($pages as $page) {
            $meta_description = get_post_meta($page->ID, '_yoast_wpseo_metadesc', true);
            if (empty($meta_description)) {
                $issues[] = array(
                    'type' => 'missing_meta_description',
                    'page_id' => $page->ID,
                    'page_title' => $page->post_title,
                    'message' => 'Missing meta description'
                );
            }
        }
        
        wp_send_json_success($issues);
    }
    
    public function ajax_bulk_seo_optimization() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $page_ids = array_map('intval', $_POST['page_ids']);
        $optimized_count = 0;
        
        foreach ($page_ids as $page_id) {
            // Update Yoast SEO meta
            update_post_meta($page_id, '_yoast_wpseo_focuskw', get_the_title($page_id));
            update_post_meta($page_id, '_yoast_wpseo_metadesc', wp_trim_words(get_post_field('post_content', $page_id), 20));
            
            $optimized_count++;
        }
        
        wp_send_json_success("Optimized {$optimized_count} pages");
    }
    
    public function ajax_bulk_generate_pages() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $locations = array_map('sanitize_text_field', $_POST['locations']);
        $business_name = sanitize_text_field($_POST['business_name']);
        $service = sanitize_text_field($_POST['service']);
        $phone = sanitize_text_field($_POST['phone']);
        $email = sanitize_email($_POST['email']);
        $website = esc_url_raw($_POST['website']);
        
        $generated_count = 0;
        
        foreach ($locations as $location) {
            $title = "{$service} in {$location} - {$business_name}";
            $content = $this->generate_page_content($business_name, $location, $service, $phone, $email, $website);
            
            $post_id = wp_insert_post(array(
                'post_title' => $title,
                'post_content' => $content,
                'post_status' => 'publish',
                'post_type' => 'page'
            ));
            
            if ($post_id) {
                $generated_count++;
            }
        }
        
        wp_send_json_success("Generated {$generated_count} pages");
    }
    
    public function ajax_import_locations_csv() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        if (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
            wp_send_json_error('No file uploaded');
        }
        
        $csv_file = $_FILES['csv_file']['tmp_name'];
        $csv_data = array_map('str_getcsv', file($csv_file));
        
        $imported_count = 0;
        
        foreach ($csv_data as $row) {
            if (count($row) >= 3) {
                $business_name = sanitize_text_field($row[0]);
                $location = sanitize_text_field($row[1]);
                $service = sanitize_text_field($row[2]);
                $phone = isset($row[3]) ? sanitize_text_field($row[3]) : '';
                $email = isset($row[4]) ? sanitize_email($row[4]) : '';
                $website = isset($row[5]) ? esc_url_raw($row[5]) : '';
                
                $title = "{$service} in {$location} - {$business_name}";
                $content = $this->generate_page_content($business_name, $location, $service, $phone, $email, $website);
                
                $post_id = wp_insert_post(array(
                    'post_title' => $title,
                    'post_content' => $content,
                    'post_status' => 'publish',
                    'post_type' => 'page'
                ));
                
                if ($post_id) {
                    $imported_count++;
                }
            }
        }
        
        wp_send_json_success("Imported {$imported_count} locations");
    }
    
    public function ajax_save_template() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $template_name = sanitize_text_field($_POST['template_name']);
        $template_content = wp_kses_post($_POST['template_content']);
        
        $templates = get_option('seo_lead_pages_templates', array());
        $templates[$template_name] = $template_content;
        
        update_option('seo_lead_pages_templates', $templates);
        
        wp_send_json_success('Template saved successfully');
    }
    
    public function ajax_delete_template() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $template_name = sanitize_text_field($_POST['template_name']);
        
        $templates = get_option('seo_lead_pages_templates', array());
        unset($templates[$template_name]);
        
        update_option('seo_lead_pages_templates', $templates);
        
        wp_send_json_success('Template deleted successfully');
    }
    
    public function ajax_get_template_library() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $templates = array(
            'wine_tasting' => array(
                'name' => 'Wine Tasting',
                'content' => 'Welcome to our wine tasting experience in {location}. We offer the finest wines and tastings for {service}. Contact us at {phone} or {email}.'
            )
        );
        
        wp_send_json_success($templates);
    }
    
    // Helper function to create missing database table
    private function create_missing_database_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            province varchar(255) NOT NULL,
            name varchar(255) NOT NULL,
            population int(11) DEFAULT 0,
            landmarks text,
            country varchar(255) DEFAULT 'Netherlands',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY province (province),
            KEY name (name)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        error_log('GEO SEO: Database table creation attempted');
    }
    
    // Location Management AJAX Handlers
    public function ajax_add_location_manually() {
        // Debug logging
        error_log('GEO SEO: ajax_add_location_manually called');
        error_log('GEO SEO: POST data: ' . print_r($_POST, true));
        
        if (!wp_verify_nonce($_POST['nonce'], 'seo_lead_pages_nonce')) {
            error_log('GEO SEO: Nonce verification failed');
            error_log('GEO SEO: Expected nonce: ' . wp_create_nonce('seo_lead_pages_nonce'));
            error_log('GEO SEO: Received nonce: ' . $_POST['nonce']);
            wp_send_json_error('Security check failed - Nonce verification failed');
        }
        
        if (!current_user_can('manage_options')) {
            error_log('GEO SEO: User does not have manage_options capability');
            wp_die('Unauthorized');
        }
        
        $city_name = sanitize_text_field($_POST['city_name']);
        $province_state = sanitize_text_field($_POST['province_state']);
        $country = sanitize_text_field($_POST['country']);
        $custom_country = sanitize_text_field($_POST['custom_country']);
        $postal_code = sanitize_text_field($_POST['postal_code']);
        $business_name = sanitize_text_field($_POST['business_name']);
        $service = sanitize_text_field($_POST['service']);
        $phone = sanitize_text_field($_POST['phone']);
        $email = sanitize_email($_POST['email']);
        $website = esc_url_raw($_POST['website']);
        
        // Handle custom country
        if ($country === 'custom') {
            if (empty($custom_country)) {
                wp_send_json_error('Please enter a custom country name');
            }
            $country = $custom_country;
        }
        
        // Validate required fields
        if (empty($city_name) || empty($province_state) || empty($country)) {
            wp_send_json_error('City name, province/state, and country are required');
        }
        
        // Create location data
        $location_data = array(
            'city_name' => $city_name,
            'province_state' => $province_state,
            'country' => $country,
            'postal_code' => $postal_code,
            'business_name' => $business_name,
            'service' => $service,
            'phone' => $phone,
            'email' => $email,
            'website' => $website,
            'created_at' => current_time('mysql')
        );
        
        // Save to options
        $saved_locations = get_option('seo_lead_pages_saved_locations', array());
        $saved_locations[] = $location_data;
        $result = update_option('seo_lead_pages_saved_locations', $saved_locations);
        
        error_log('GEO SEO: Location data saved: ' . print_r($location_data, true));
        error_log('GEO SEO: Update option result: ' . ($result ? 'true' : 'false'));
        
        error_log('GEO SEO: Sending success response');
        wp_send_json_success(array('message' => 'Location added successfully!'));
        error_log('GEO SEO: Response sent successfully');
    }
    
    public function ajax_add_location_and_generate_page() {
        // Debug logging
        error_log('GEO SEO: ajax_add_location_and_generate_page called');
        error_log('GEO SEO: POST data: ' . print_r($_POST, true));
        
        if (!wp_verify_nonce($_POST['nonce'], 'seo_lead_pages_nonce')) {
            error_log('GEO SEO: Nonce verification failed in add_location_and_generate_page');
            error_log('GEO SEO: Expected nonce: ' . wp_create_nonce('seo_lead_pages_nonce'));
            error_log('GEO SEO: Received nonce: ' . $_POST['nonce']);
            wp_send_json_error('Security check failed - Nonce verification failed');
        }
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $city_name = sanitize_text_field($_POST['city_name']);
        $province_state = sanitize_text_field($_POST['province_state']);
        $country = sanitize_text_field($_POST['country']);
        $custom_country = sanitize_text_field($_POST['custom_country']);
        $postal_code = sanitize_text_field($_POST['postal_code']);
        $business_name = sanitize_text_field($_POST['business_name']);
        $service = sanitize_text_field($_POST['service']);
        $phone = sanitize_text_field($_POST['phone']);
        $email = sanitize_email($_POST['email']);
        $website = esc_url_raw($_POST['website']);
        
        // Handle custom country
        if ($country === 'custom') {
            if (empty($custom_country)) {
                wp_send_json_error('Please enter a custom country name');
            }
            $country = $custom_country;
        }
        
        // Validate required fields
        if (empty($city_name) || empty($province_state) || empty($country)) {
            wp_send_json_error('City name, province/state, and country are required');
        }
        
        // Create location data
        $location_data = array(
            'city_name' => $city_name,
            'province_state' => $province_state,
            'country' => $country,
            'postal_code' => $postal_code,
            'business_name' => $business_name,
            'service' => $service,
            'phone' => $phone,
            'email' => $email,
            'website' => $website,
            'created_at' => current_time('mysql')
        );
        
        // Save to options
        $saved_locations = get_option('seo_lead_pages_saved_locations', array());
        $saved_locations[] = $location_data;
        update_option('seo_lead_pages_saved_locations', $saved_locations);
        
        // Generate page if business details are provided
        if (!empty($business_name) && !empty($service)) {
            $location = $city_name . ', ' . $province_state;
            $title = "{$service} in {$location} - {$business_name}";
            $content = $this->generate_page_content($business_name, $location, $service, $phone, $email, $website);
            
            $post_id = wp_insert_post(array(
                'post_title' => $title,
                'post_content' => $content,
                'post_status' => 'publish',
                'post_type' => 'page'
            ));
            
            if ($post_id) {
                error_log('GEO SEO: Location and page generated successfully, post_id: ' . $post_id);
                wp_send_json_success(array('message' => 'Location added and page generated successfully!', 'post_id' => $post_id));
            } else {
                error_log('GEO SEO: Location added but page generation failed');
                wp_send_json_success(array('message' => 'Location added, but failed to generate page.'));
            }
        } else {
            error_log('GEO SEO: Location added without page generation');
            wp_send_json_success(array('message' => 'Location added successfully!'));
        }
    }
    
    public function ajax_get_saved_locations() {
        // Debug logging
        error_log('GEO SEO: ajax_get_saved_locations called');
        error_log('GEO SEO: POST data: ' . print_r($_POST, true));
        
        if (!wp_verify_nonce($_POST['nonce'], 'seo_lead_pages_nonce')) {
            error_log('GEO SEO: Nonce verification failed in get_saved_locations');
            wp_send_json_error('Security check failed');
        }
        
        if (!current_user_can('manage_options')) {
            error_log('GEO SEO: User does not have manage_options capability in get_saved_locations');
            wp_die('Unauthorized');
        }
        
        // Get manually saved locations from options
        $saved_locations = get_option('seo_lead_pages_saved_locations', array());
        
        // Get imported locations from database
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'");
        if (!$table_exists) {
            error_log('GEO SEO: Table ' . $table_name . ' does not exist');
            $imported_locations = array();
        } else {
            $imported_locations = $wpdb->get_results("SELECT * FROM $table_name ORDER BY created_at DESC", ARRAY_A);
            if ($wpdb->last_error) {
                error_log('GEO SEO: Database error in get_saved_locations: ' . $wpdb->last_error);
                $imported_locations = array();
            }
        }
        
        error_log('GEO SEO: Found ' . count($saved_locations) . ' saved locations and ' . count($imported_locations) . ' imported locations');
        
        // If table doesn't exist, try to create it
        if (!$table_exists) {
            error_log('GEO SEO: Attempting to create missing database table');
            $this->create_missing_database_table();
        }
        
        if (empty($saved_locations) && empty($imported_locations)) {
            wp_send_json_success('<p>No saved locations yet. Add your first location above or import from a country!</p>');
        }
        
        $html = '<table class="wp-list-table widefat fixed striped">';
        $html .= '<thead><tr>';
        $html .= '<th>City</th><th>Province/State</th><th>Country</th><th>Business</th><th>Service</th><th>Date Added</th><th>Actions</th>';
        $html .= '</tr></thead><tbody>';
        
        // Show imported locations first
        foreach ($imported_locations as $location) {
            $html .= '<tr>';
            $html .= '<td>' . esc_html($location['name']) . '</td>';
            $html .= '<td>' . esc_html($location['province']) . '</td>';
            $html .= '<td>' . esc_html($location['country']) . '</td>';
            $html .= '<td>Imported</td>';
            $html .= '<td>N/A</td>';
            $html .= '<td>' . date('Y-m-d', strtotime($location['created_at'])) . '</td>';
            $html .= '<td>';
            $html .= '<button type="button" class="button button-small delete-imported-location" data-location-id="' . $location['id'] . '">Delete</button>';
            $html .= '</td>';
            $html .= '</tr>';
        }
        
        // Show manually saved locations
        foreach ($saved_locations as $index => $location) {
            $html .= '<tr>';
            $html .= '<td>' . esc_html($location['city_name']) . '</td>';
            $html .= '<td>' . esc_html($location['province_state']) . '</td>';
            $html .= '<td>' . esc_html($location['country']) . '</td>';
            $html .= '<td>' . esc_html($location['business_name']) . '</td>';
            $html .= '<td>' . esc_html($location['service']) . '</td>';
            $html .= '<td>' . date('Y-m-d', strtotime($location['created_at'])) . '</td>';
            $html .= '<td>';
            $html .= '<button type="button" class="button button-small delete-location" data-location-index="' . $index . '">Delete</button>';
            $html .= '</td>';
            $html .= '</tr>';
        }
        
        $html .= '</tbody></table>';
        
        wp_send_json_success($html);
    }
    
    // Page Generator AJAX Handlers
    public function ajax_quick_generate_pages() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        // Check if required shortcode settings are configured
        $shortcode_mappings = get_option('seo_lead_pages_shortcode_mappings', array());
        $required_shortcodes = array('service_name', 'business_name', 'phone_number', 'contact_email');
        $missing_shortcodes = array();
        
        foreach ($required_shortcodes as $shortcode) {
            if (empty($shortcode_mappings[$shortcode])) {
                $missing_shortcodes[] = $shortcode;
            }
        }
        
        if (!empty($missing_shortcodes)) {
            wp_send_json_error(array(
                'message' => 'Shortcode instellingen vereist! Configureer eerst: ' . implode(', ', $missing_shortcodes),
                'redirect_url' => admin_url('admin.php?page=seo-lead-pages-shortcodes')
            ));
        }
        
        $selected_location = sanitize_text_field($_POST['selected_location']);
        $template = sanitize_text_field($_POST['template']);
        $url_logic = sanitize_text_field($_POST['url_logic'] ?? 'service_in_location');
        $custom_slug_template = sanitize_text_field($_POST['custom_slug_template'] ?? '');
        $generate_multiple = isset($_POST['generate_multiple']) && $_POST['generate_multiple'] === 'on';
        
        // Debug logging
        error_log('GEO Local SEO Debug - Quick Generate:');
        error_log('Selected Location: ' . $selected_location);
        error_log('Template: ' . $template);
        error_log('Generate Multiple: ' . ($generate_multiple ? 'true' : 'false'));
        error_log('Generate Multiple Raw: ' . (isset($_POST['generate_multiple']) ? $_POST['generate_multiple'] : 'not set'));
        
        $saved_locations = get_option('seo_lead_pages_saved_locations', array());
        error_log('Saved Locations Count: ' . count($saved_locations));
        
        error_log('About to check generate_multiple condition...');
        
        if ($generate_multiple) {
            error_log('Entering generate_multiple = true branch');
            // Generate pages for all locations (both manually saved and imported)
            $generated_count = 0;
            
            // Get imported locations from database
            global $wpdb;
            $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
            $imported_locations = $wpdb->get_results("SELECT * FROM $table_name ORDER BY name ASC", ARRAY_A);
            
            // Process manually saved locations
            foreach ($saved_locations as $location) {
                if (!empty($location['business_name']) && !empty($location['service'])) {
                    $title_slug = $this->generate_page_title_and_slug($location['service'], $location, $url_logic, $custom_slug_template);
                    $title = $title_slug['title'];
                    
                    // Use template if provided, otherwise use default content
                    if (!empty($template)) {
                        $content = $this->generate_content_from_template($template, $location);
                    } else {
                        $content = $this->generate_page_content($location['business_name'], $location_str, $location['service'], $location['phone'], $location['email'], $location['website']);
                    }
                    
                    $post_id = wp_insert_post(array(
                        'post_title' => $title,
                        'post_name' => $title_slug['slug'],
                        'post_content' => $content,
                        'post_status' => 'publish',
                        'post_type' => 'page'
                    ));
                    
                    if ($post_id) {
                        $generated_count++;
                    }
                }
            }
            
            // Process imported locations
            foreach ($imported_locations as $imported_location) {
                // Convert imported location to expected format
                $location = array(
                    'city_name' => $imported_location['name'],
                    'province_state' => $imported_location['province'],
                    'country' => $imported_location['country'],
                    'business_name' => get_option('seo_lead_pages_business_name', 'StormSEO'),
                    'service' => 'SEO',
                    'phone' => get_option('seo_lead_pages_phone_number', ''),
                    'email' => get_option('seo_lead_pages_contact_email', get_option('admin_email')),
                    'website' => get_option('seo_lead_pages_website_url', '')
                );
                
                $title_slug = $this->generate_page_title_and_slug('SEO', $location, $url_logic, $custom_slug_template);
                $title = $title_slug['title'];
                $content = $this->generate_page_content($location['business_name'], $imported_location['name'] . ', ' . $imported_location['province'], 'SEO', '', '', '');
                
                $post_id = wp_insert_post(array(
                    'post_title' => $title,
                    'post_name' => $title_slug['slug'],
                    'post_content' => $content,
                    'post_status' => 'publish',
                    'post_type' => 'page'
                ));
                
                if ($post_id) {
                    $generated_count++;
                }
            }
            
            wp_send_json_success(array('message' => "Generated {$generated_count} pages successfully!"));
        } else {
            error_log('Entering generate_multiple = false branch');
            // Generate page for selected location
            try {
                error_log('Starting try block...');
                if (empty($selected_location) && $selected_location !== '0') {
                    error_log('Selected location is empty');
                    wp_send_json_error('Please select a location');
                }
                
                error_log('Selected location is not empty: ' . $selected_location);
                
                // Check if selected_location is a valid index or imported location
                $location = null;
                if (strpos($selected_location, 'imported_') === 0) {
                    // Handle imported location
                    $location_id = str_replace('imported_', '', $selected_location);
                    global $wpdb;
                    $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
                    $imported_location = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $location_id), ARRAY_A);
                    
                    if ($imported_location) {
                        // Convert imported location to the expected format
                        $location = array(
                            'city_name' => $imported_location['name'],
                            'province_state' => $imported_location['province'],
                            'country' => $imported_location['country'],
                            'business_name' => get_option('seo_lead_pages_business_name', 'StormSEO'),
                            'service' => 'SEO', // Default service for imported locations
                            'phone' => get_option('seo_lead_pages_phone_number', ''),
                            'email' => get_option('seo_lead_pages_contact_email', get_option('admin_email')),
                            'website' => get_option('seo_lead_pages_website_url', '')
                        );
                        error_log('Using imported location: ' . print_r($location, true));
                    } else {
                        error_log('Imported location not found: ' . $location_id);
                        wp_send_json_error('Imported location not found');
                    }
                } else if (is_numeric($selected_location) && isset($saved_locations[$selected_location])) {
                    // Handle manually saved location
                    $location = $saved_locations[$selected_location];
                    error_log('Using manually saved location: ' . print_r($location, true));
                } else {
                    error_log('Invalid location index: ' . $selected_location);
                    wp_send_json_error('Please select a valid location');
                }
                
                // Debug: Log location data
                error_log('Location Data: ' . print_r($location, true));
                
                // Check if location has required fields and add defaults if missing
                error_log('Checking required fields...');
                if (strpos($selected_location, 'imported_') !== 0) {
                    // Add default values if business_name or service is empty
                    if (empty($location['business_name'])) {
                        $location['business_name'] = get_option('seo_lead_pages_business_name', get_option('blogname', 'Your Business'));
                        error_log('Added default business_name: ' . $location['business_name']);
                    }
                    if (empty($location['service'])) {
                        $location['service'] = get_option('seo_lead_pages_default_service', 'SEO Services');
                        error_log('Added default service: ' . $location['service']);
                    }
                }
                
                error_log('Required fields check passed, creating location string...');
                $title_slug = $this->generate_page_title_and_slug($location['service'], $location, $url_logic, $custom_slug_template);
                $title = $title_slug['title'];
                
                // Debug: Log title and location string
                error_log('Generated Title: ' . $title);
                error_log('Location String: ' . $location_str);
                
                // Check if generate_page_content method exists
                error_log('Checking if generate_page_content method exists...');
                if (!method_exists($this, 'generate_page_content')) {
                    error_log('ERROR: generate_page_content method does not exist');
                    wp_send_json_error('Page generation method not found');
                }
                
                error_log('Method exists, generating content...');
                
                // Use template if provided, otherwise use default content
                if (!empty($template)) {
                    error_log('Using template: ' . $template);
                    $content = $this->generate_content_from_template($template, $location);
                } else {
                    error_log('No template provided, using default content generation');
                    $content = $this->generate_page_content($location['business_name'], $location_str, $location['service'], $location['phone'], $location['email'], $location['website']);
                }
                
                // Debug: Log content length
                error_log('Generated Content Length: ' . strlen($content));
                
                error_log('About to create WordPress post...');
                $post_id = wp_insert_post(array(
                    'post_title' => $title,
                    'post_name' => $title_slug['slug'],
                    'post_content' => $content,
                    'post_status' => 'publish',
                    'post_type' => 'page'
                ));
                
                // Debug: Log post creation result
                error_log('Post creation result - Post ID: ' . $post_id . ', Title: ' . $title);
                
                if ($post_id) {
                    error_log('Post created successfully!');
                    wp_send_json_success(array('message' => 'Page generated successfully!', 'post_id' => $post_id));
                } else {
                    error_log('Failed to create post - wp_insert_post returned: ' . $post_id);
                    wp_send_json_error('Failed to generate page');
                }
            } catch (Exception $e) {
                error_log('PHP Error in ajax_quick_generate_pages: ' . $e->getMessage());
                wp_send_json_error('Error generating page: ' . $e->getMessage());
            }
        }
    }
    
    
    public function ajax_get_saved_locations_for_generator() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        // Get manually saved locations from options
        $saved_locations = get_option('seo_lead_pages_saved_locations', array());
        
        // Get imported locations from database
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
        $imported_locations = $wpdb->get_results("SELECT * FROM $table_name ORDER BY name ASC", ARRAY_A);
        
        // Debug: Log the locations count
        error_log('GEO Local SEO Debug: Found ' . count($saved_locations) . ' manually saved locations and ' . count($imported_locations) . ' imported locations');
        
        if (empty($saved_locations) && empty($imported_locations)) {
            wp_send_json_success('<option value="">No saved locations available</option>');
        }
        
        $html = '<option value="">Choose a saved location...</option>';
        
        // Add imported locations first
        foreach ($imported_locations as $location) {
            $location_str = $location['name'] . ', ' . $location['province'] . ', ' . $location['country'];
            $html .= '<option value="imported_' . $location['id'] . '">' . esc_html($location_str) . ' (Imported)</option>';
        }
        
        // Add manually saved locations
        foreach ($saved_locations as $index => $location) {
            $location_str = $location['city_name'] . ', ' . $location['province_state'] . ', ' . $location['country'];
            $html .= '<option value="' . $index . '">' . esc_html($location_str) . '</option>';
        }
        
        wp_send_json_success($html);
    }
    
    public function ajax_get_saved_templates_for_generator() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $templates = get_option('seo_lead_pages_templates', array());
        
        if (empty($templates)) {
            wp_send_json_success('<option value="">No saved templates available</option>');
        }
        
        $html = '<option value="">Use default template</option>';
        
        foreach ($templates as $name => $template) {
            $html .= '<option value="' . esc_attr($name) . '">' . esc_html($name) . '</option>';
        }
        
        wp_send_json_success($html);
    }
    
    public function ajax_get_templates_overview() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $templates = get_option('seo_lead_pages_templates', array());
        
        // Debug logging
        error_log('GEO Local SEO Debug - Templates from options: ' . print_r($templates, true));
        
        $template_array = array();
        foreach ($templates as $name => $content) {
            $template_array[] = array(
                'name' => $name,
                'content' => $content
            );
        }
        
        error_log('GEO Local SEO Debug - Template array: ' . print_r($template_array, true));
        
        wp_send_json_success(array('templates' => $template_array));
    }
    
    // Dashboard AJAX Handlers
    public function ajax_get_dashboard_stats() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        // Get saved locations count
        $saved_locations = get_option('seo_lead_pages_saved_locations', array());
        $locations_count = count($saved_locations);
        
        // Get generated pages count
        $pages = get_posts(array(
            'post_type' => 'page',
            'post_status' => 'publish',
            'meta_query' => array(
                array(
                    'key' => '_seo_lead_page',
                    'value' => '1',
                    'compare' => '='
                )
            ),
            'numberposts' => -1
        ));
        $pages_count = count($pages);
        
        // Get total leads count
        global $wpdb;
        $leads_count = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}seo_lead_pages_leads");
        
        // Get templates count
        $templates = get_option('seo_lead_pages_templates', array());
        $templates_count = count($templates);
        
        wp_send_json_success(array(
            'locations' => $locations_count,
            'pages' => $pages_count,
            'leads' => $leads_count,
            'templates' => $templates_count
        ));
    }
    
    // Additional Location Management AJAX Handlers
    public function ajax_generate_page_from_saved_location() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $location_index = $_POST['location_index'];
        $location = null;
        
        if (strpos($location_index, 'imported_') === 0) {
            // Handle imported location
            $location_id = str_replace('imported_', '', $location_index);
            global $wpdb;
            $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
            $imported_location = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $location_id), ARRAY_A);
            
            if ($imported_location) {
                // Convert imported location to the expected format
                $location = array(
                    'city_name' => $imported_location['name'],
                    'province_state' => $imported_location['province'],
                    'country' => $imported_location['country'],
                    'business_name' => 'Imported Business',
                    'service' => 'SEO',
                    'phone' => '',
                    'email' => '',
                    'website' => ''
                );
            } else {
                wp_send_json_error('Imported location not found');
            }
        } else {
            // Handle manually saved location
            $location_index = intval($location_index);
            $saved_locations = get_option('seo_lead_pages_saved_locations', array());
            
            if (!isset($saved_locations[$location_index])) {
                wp_send_json_error('Location not found');
            }
            
            $location = $saved_locations[$location_index];
            
            if (empty($location['business_name']) || empty($location['service'])) {
                wp_send_json_error('Location missing business name or service');
            }
        }
        
        $location_str = $location['city_name'] . ', ' . $location['province_state'];
        $title = "{$location['service']} in {$location_str} - {$location['business_name']}";
        $content = $this->generate_page_content($location['business_name'], $location_str, $location['service'], $location['phone'], $location['email'], $location['website']);
        
        $post_id = wp_insert_post(array(
            'post_title' => $title,
            'post_content' => $content,
            'post_status' => 'publish',
            'post_type' => 'page'
        ));
        
        if ($post_id) {
            wp_send_json_success('Page generated successfully!');
        } else {
            wp_send_json_error('Failed to generate page');
        }
    }
    
    public function ajax_delete_saved_location() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $location_index = intval($_POST['location_index']);
        $saved_locations = get_option('seo_lead_pages_saved_locations', array());
        
        if (!isset($saved_locations[$location_index])) {
            wp_send_json_error('Location not found');
        }
        
        unset($saved_locations[$location_index]);
        $saved_locations = array_values($saved_locations); // Re-index array
        
        update_option('seo_lead_pages_saved_locations', $saved_locations);
        
        wp_send_json_success('Location deleted successfully!');
    }
    
    public function ajax_delete_imported_location() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $location_id = intval($_POST['location_id']);
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
        
        $result = $wpdb->delete($table_name, array('id' => $location_id), array('%d'));
        
        if ($result === false) {
            wp_send_json_error('Failed to delete location: ' . $wpdb->last_error);
        }
        
        wp_send_json_success('Imported location deleted successfully!');
    }
    
    public function ajax_get_bulk_delete_countries() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'");
        if (!$table_exists) {
            wp_send_json_success('<option value="">No countries available</option>');
            return;
        }
        
        $countries = $wpdb->get_results("SELECT DISTINCT country FROM $table_name ORDER BY country ASC");
        
        if ($wpdb->last_error) {
            error_log('GEO SEO: Database error in get_bulk_delete_countries: ' . $wpdb->last_error);
            wp_send_json_success('<option value="">Error loading countries</option>');
            return;
        }
        
        $html = '<option value="">Select Country</option>';
        foreach ($countries as $country) {
            $html .= '<option value="' . esc_attr($country->country) . '">' . esc_html($country->country) . '</option>';
        }
        
        wp_send_json_success($html);
    }
    
    public function ajax_get_bulk_delete_provinces() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'");
        if (!$table_exists) {
            wp_send_json_success('<option value="">No provinces available</option>');
            return;
        }
        
        $provinces = $wpdb->get_results("SELECT DISTINCT province FROM $table_name ORDER BY province ASC");
        
        if ($wpdb->last_error) {
            error_log('GEO SEO: Database error in get_bulk_delete_provinces: ' . $wpdb->last_error);
            wp_send_json_success('<option value="">Error loading provinces</option>');
            return;
        }
        
        $html = '<option value="">Select Province/State</option>';
        foreach ($provinces as $province) {
            $html .= '<option value="' . esc_attr($province->province) . '">' . esc_html($province->province) . '</option>';
        }
        
        wp_send_json_success($html);
    }
    
    public function ajax_get_bulk_delete_count() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $country = sanitize_text_field($_POST['country']);
        $province = sanitize_text_field($_POST['province']);
        $delete_type = sanitize_text_field($_POST['delete_type']);
        
        $count = 0;
        
        if ($delete_type === 'imported' || $delete_type === 'both') {
            global $wpdb;
            $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
            
            $where_conditions = array();
            $where_values = array();
            
            if ($country) {
                $where_conditions[] = 'country = %s';
                $where_values[] = $country;
            }
            
            if ($province) {
                $where_conditions[] = 'province = %s';
                $where_values[] = $province;
            }
            
            if (!empty($where_conditions)) {
                $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
                $count += $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table_name $where_clause", $where_values));
            }
        }
        
        if ($delete_type === 'manual' || $delete_type === 'both') {
            $saved_locations = get_option('seo_lead_pages_saved_locations', array());
            
            foreach ($saved_locations as $location) {
                $matches = true;
                
                if ($country && $location['country'] !== $country) {
                    $matches = false;
                }
                
                if ($province && $location['province_state'] !== $province) {
                    $matches = false;
                }
                
                if ($matches) {
                    $count++;
                }
            }
        }
        
        wp_send_json_success(array('count' => $count));
    }
    
    public function ajax_bulk_delete_locations() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $country = sanitize_text_field($_POST['country']);
        $province = sanitize_text_field($_POST['province']);
        $delete_type = sanitize_text_field($_POST['delete_type']);
        
        $deleted_count = 0;
        
        if ($delete_type === 'imported' || $delete_type === 'both') {
            global $wpdb;
            $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
            
            $where_conditions = array();
            $where_values = array();
            
            if ($country) {
                $where_conditions[] = 'country = %s';
                $where_values[] = $country;
            }
            
            if ($province) {
                $where_conditions[] = 'province = %s';
                $where_values[] = $province;
            }
            
            if (!empty($where_conditions)) {
                $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
                $deleted_count += $wpdb->query($wpdb->prepare("DELETE FROM $table_name $where_clause", $where_values));
            }
        }
        
        if ($delete_type === 'manual' || $delete_type === 'both') {
            $saved_locations = get_option('seo_lead_pages_saved_locations', array());
            $filtered_locations = array();
            
            foreach ($saved_locations as $location) {
                $matches = true;
                
                if ($country && $location['country'] !== $country) {
                    $matches = false;
                }
                
                if ($province && $location['province_state'] !== $province) {
                    $matches = false;
                }
                
                if (!$matches) {
                    $filtered_locations[] = $location;
                } else {
                    $deleted_count++;
                }
            }
            
            update_option('seo_lead_pages_saved_locations', $filtered_locations);
        }
        
        $message = "Successfully deleted {$deleted_count} locations";
        if ($country) {
            $message .= " from {$country}";
        }
        if ($province) {
            $message .= " in {$province}";
        }
        
        wp_send_json_success(array('message' => $message, 'deleted_count' => $deleted_count));
    }
    
    public function ajax_get_saved_locations_for_bulk_delete() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $country = sanitize_text_field($_POST['country'] ?? '');
        $province = sanitize_text_field($_POST['province'] ?? '');
        
        // Get manually saved locations from options
        $saved_locations = get_option('seo_lead_pages_saved_locations', array());
        
        // Get imported locations from database
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
        
        $where_conditions = array();
        $where_values = array();
        
        if ($country) {
            $where_conditions[] = "country = %s";
            $where_values[] = $country;
        }
        
        if ($province) {
            $where_conditions[] = "province = %s";
            $where_values[] = $province;
        }
        
        $where_clause = '';
        if (!empty($where_conditions)) {
            $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
        }
        
        $query = "SELECT * FROM $table_name $where_clause ORDER BY created_at DESC";
        if (!empty($where_values)) {
            $imported_locations = $wpdb->get_results($wpdb->prepare($query, $where_values), ARRAY_A);
        } else {
            $imported_locations = $wpdb->get_results($query, ARRAY_A);
        }
        
        $html = '';
        
        // Add imported locations
        foreach ($imported_locations as $location) {
            if (!$country || $location['country'] === $country) {
                if (!$province || $location['province'] === $province) {
                    $html .= '<option value="imported_' . $location['id'] . '">' . esc_html($location['name'] . ', ' . $location['province'] . ', ' . $location['country']) . ' (Imported)</option>';
                }
            }
        }
        
        // Add manually saved locations
        foreach ($saved_locations as $index => $location) {
            if (!$country || $location['country'] === $country) {
                if (!$province || $location['province_state'] === $province) {
                    $html .= '<option value="manual_' . $index . '">' . esc_html($location['city_name'] . ', ' . $location['province_state'] . ', ' . $location['country']) . ' (Manual)</option>';
                }
            }
        }
        
        if (empty($html)) {
            $html = '<option value="">No locations found matching criteria</option>';
        }
        
        wp_send_json_success($html);
    }
    
    public function ajax_bulk_delete_saved_locations() {
        check_ajax_referer('seo_lead_pages_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $locations = array_map('sanitize_text_field', $_POST['locations']);
        $deleted_count = 0;
        
        // Get manually saved locations from options
        $saved_locations = get_option('seo_lead_pages_saved_locations', array());
        
        foreach ($locations as $location) {
            if (strpos($location, 'imported_') === 0) {
                // Delete imported location
                $location_id = str_replace('imported_', '', $location);
                global $wpdb;
                $table_name = $wpdb->prefix . 'seo_lead_pages_locations';
                $result = $wpdb->delete($table_name, array('id' => $location_id), array('%d'));
                if ($result !== false) {
                    $deleted_count++;
                }
            } elseif (strpos($location, 'manual_') === 0) {
                // Delete manually saved location
                $location_index = str_replace('manual_', '', $location);
                if (isset($saved_locations[$location_index])) {
                    unset($saved_locations[$location_index]);
                    $deleted_count++;
                }
            }
        }
        
        // Update the saved locations option
        update_option('seo_lead_pages_saved_locations', array_values($saved_locations));
        
        $message = "Successfully deleted {$deleted_count} locations.";
        wp_send_json_success(array('message' => $message, 'deleted_count' => $deleted_count));
    }
    
    public function lead_management_page() {
        // Ensure premium status is checked when page is loaded
        $this->check_premium_status();
        ?>
        <div class="wrap">
            <h1>📋 Lead Management</h1>
            <p>Manage and track leads from your landing pages. Update status, delete leads, and export data.</p>
            
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <h3>📊 Lead Statistics</h3>
                <div id="lead-stats" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin: 20px 0;">
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; text-align: center;">
                        <h4 style="margin: 0; color: #0073aa;" id="total-leads">-</h4>
                        <p style="margin: 5px 0 0 0;">Total Leads</p>
                    </div>
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; text-align: center;">
                        <h4 style="margin: 0; color: #28a745;" id="open-leads">-</h4>
                        <p style="margin: 5px 0 0 0;">Open Leads</p>
                    </div>
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; text-align: center;">
                        <h4 style="margin: 0; color: #ffc107;" id="in-progress-leads">-</h4>
                        <p style="margin: 5px 0 0 0;">In Progress</p>
                    </div>
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; text-align: center;">
                        <h4 style="margin: 0; color: #17a2b8;" id="completed-leads">-</h4>
                        <p style="margin: 5px 0 0 0;">Completed</p>
                    </div>
                </div>
            </div>
            
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin: 20px 0;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                    <h3 style="margin: 0;">📋 All Leads</h3>
                    <div>
                        <button id="export-leads" class="button button-secondary">📥 Export CSV</button>
                        <button id="bulk-delete-leads" class="button button-secondary" style="background: #dc3545; color: white;">🗑️ Bulk Delete</button>
                    </div>
                </div>
                
                <div id="leads-table-container">
                    <p>Loading leads...</p>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Load lead statistics
            function loadLeadStats() {
                $.post(ajaxurl, {
                    action: 'get_lead_stats',
                    nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        $('#total-leads').text(response.data.total);
                        $('#open-leads').text(response.data.open);
                        $('#in-progress-leads').text(response.data.in_progress);
                        $('#completed-leads').text(response.data.completed);
                    }
                });
            }
            
            // Load leads table
            function loadLeadsTable() {
                $.post(ajaxurl, {
                    action: 'get_leads_table',
                    nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        $('#leads-table-container').html(response.data);
                    }
                });
            }
            
            // Update lead status
            $(document).on('change', '.lead-status-select', function() {
                var leadId = $(this).data('lead-id');
                var newStatus = $(this).val();
                
                $.post(ajaxurl, {
                    action: 'update_lead_status',
                    lead_id: leadId,
                    status: newStatus,
                    nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        loadLeadStats();
                        loadLeadsTable();
                    } else {
                        alert('Error updating lead status: ' + response.data);
                    }
                });
            });
            
            // Delete single lead
            $(document).on('click', '.delete-lead', function() {
                if (confirm('Are you sure you want to delete this lead?')) {
                    var leadId = $(this).data('lead-id');
                    
                    $.post(ajaxurl, {
                        action: 'delete_lead',
                        lead_id: leadId,
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    }, function(response) {
                        if (response.success) {
                            loadLeadStats();
                            loadLeadsTable();
                        } else {
                            alert('Error deleting lead: ' + response.data);
                        }
                    });
                }
            });
            
            // Bulk delete leads
            $('#bulk-delete-leads').on('click', function() {
                var selectedLeads = [];
                $('.lead-checkbox:checked').each(function() {
                    selectedLeads.push($(this).val());
                });
                
                if (selectedLeads.length === 0) {
                    alert('Please select leads to delete.');
                    return;
                }
                
                if (confirm('Are you sure you want to delete ' + selectedLeads.length + ' selected leads?')) {
                    $.post(ajaxurl, {
                        action: 'bulk_delete_leads',
                        lead_ids: selectedLeads,
                        nonce: '<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>'
                    }, function(response) {
                        if (response.success) {
                            loadLeadStats();
                            loadLeadsTable();
                        } else {
                            alert('Error deleting leads: ' + response.data);
                        }
                    });
                }
            });
            
            // Export leads
            $('#export-leads').on('click', function() {
                window.location.href = ajaxurl + '?action=export_leads_csv&nonce=<?php echo wp_create_nonce('seo_lead_pages_nonce'); ?>';
            });
            
            // Initial load
            loadLeadStats();
            loadLeadsTable();
        });
        </script>
        <?php
    }
    
    public function on_premium_loaded() {
        // Premium addon is loaded, enable premium features
        $this->premium_active = true;
        
        // Add premium filters and actions
        add_filter('seo_lead_pages_get_available_templates', array($this, 'get_available_templates'));
        add_filter('seo_lead_pages_is_premium_active', array($this, 'is_premium_active'));
        
        // Remove premium addon notice since premium is now active
        remove_action('admin_notices', array($this, 'premium_addon_notice'));
    }
    
    public function is_premium_active() {
        // Double check premium status
        $this->check_premium_status();
        
        // Double check that property exists
        if (!isset($this->premium_active)) {
            $this->premium_active = false;
        }
        
        return $this->premium_active;
    }
    
    private function get_premium_active() {
        // Always check premium status before returning
        $this->check_premium_status();
        
        // Double check that property exists
        if (!isset($this->premium_active)) {
            $this->premium_active = false;
        }
        
        return $this->premium_active;
    }
    
    public function get_available_templates($templates) {
        if ($this->get_premium_active()) {
            // Add premium templates
            $premium_templates = array(
                'Professional SEO' => 'Professional SEO template with advanced features',
                'E-commerce SEO' => 'E-commerce focused SEO template',
                'Local Business' => 'Local business optimization template',
                'Service Provider' => 'Service provider focused template',
                'Multi-location' => 'Multi-location business template'
            );
            
            return array_merge($templates, $premium_templates);
        }
        
        return $templates;
    }
    
    public function form_settings_page() {
        // Ensure premium status is checked when page is loaded
        $this->check_premium_status();
        // Handle form submission
        if (isset($_POST['submit']) && wp_verify_nonce($_POST['form_settings_nonce'], 'form_settings_action')) {
            update_option('seo_lead_pages_form_title', sanitize_text_field($_POST['form_title']));
            update_option('seo_lead_pages_contact_label', sanitize_text_field($_POST['contact_label']));
            update_option('seo_lead_pages_name_label', sanitize_text_field($_POST['name_label']));
            update_option('seo_lead_pages_email_label', sanitize_text_field($_POST['email_label']));
            update_option('seo_lead_pages_phone_label', sanitize_text_field($_POST['phone_label']));
            update_option('seo_lead_pages_company_label', sanitize_text_field($_POST['company_label']));
            update_option('seo_lead_pages_message_label', sanitize_text_field($_POST['message_label']));
            update_option('seo_lead_pages_message_placeholder', sanitize_textarea_field($_POST['message_placeholder']));
            update_option('seo_lead_pages_submit_button', sanitize_text_field($_POST['submit_button']));
            
            // Contact display options
            update_option('seo_lead_pages_show_phone', isset($_POST['show_phone']) ? 1 : 0);
            update_option('seo_lead_pages_show_email', isset($_POST['show_email']) ? 1 : 0);
            update_option('seo_lead_pages_show_website', isset($_POST['show_website']) ? 1 : 0);
            
            // Form styling options
            update_option('seo_lead_pages_form_background_color', sanitize_text_field($_POST['form_background_color']));
            update_option('seo_lead_pages_form_border_color', sanitize_text_field($_POST['form_border_color']));
            update_option('seo_lead_pages_form_text_color', sanitize_text_field($_POST['form_text_color']));
            update_option('seo_lead_pages_form_title_color', sanitize_text_field($_POST['form_title_color']));
            update_option('seo_lead_pages_input_background_color', sanitize_text_field($_POST['input_background_color']));
            update_option('seo_lead_pages_input_border_color', sanitize_text_field($_POST['input_border_color']));
            update_option('seo_lead_pages_input_text_color', sanitize_text_field($_POST['input_text_color']));
            update_option('seo_lead_pages_button_background_color', sanitize_text_field($_POST['button_background_color']));
            update_option('seo_lead_pages_button_text_color', sanitize_text_field($_POST['button_text_color']));
            update_option('seo_lead_pages_button_hover_color', sanitize_text_field($_POST['button_hover_color']));
            update_option('seo_lead_pages_form_border_radius', intval($_POST['form_border_radius']));
            update_option('seo_lead_pages_form_padding', intval($_POST['form_padding']));
            update_option('seo_lead_pages_form_width', sanitize_text_field($_POST['form_width']));
            update_option('seo_lead_pages_form_alignment', sanitize_text_field($_POST['form_alignment']));
            
            // Debug: Log the saved values
            error_log('GEO Local SEO Debug - Form styling saved:');
            error_log('Background: ' . get_option('seo_lead_pages_form_background_color'));
            error_log('Button: ' . get_option('seo_lead_pages_button_background_color'));
            
            echo '<div class="notice notice-success"><p>Form settings saved successfully!</p></div>';
        }
        
        // Get current settings
        $form_title = get_option('seo_lead_pages_form_title', __('Free SEO Advice for', 'geo-local-seo-generator'));
        $contact_label = get_option('seo_lead_pages_contact_label', __('Direct contact:', 'geo-local-seo-generator'));
        $name_label = get_option('seo_lead_pages_name_label', __('Name', 'geo-local-seo-generator'));
        $email_label = get_option('seo_lead_pages_email_label', __('Email address', 'geo-local-seo-generator'));
        $phone_label = get_option('seo_lead_pages_phone_label', __('Phone number', 'geo-local-seo-generator'));
        $company_label = get_option('seo_lead_pages_company_label', __('Company name', 'geo-local-seo-generator'));
        $message_label = get_option('seo_lead_pages_message_label', __('Message', 'geo-local-seo-generator'));
        $message_placeholder = get_option('seo_lead_pages_message_placeholder', __('Tell us about your SEO goals and challenges...', 'geo-local-seo-generator'));
        $submit_button = get_option('seo_lead_pages_submit_button', __('Request Free SEO Advice', 'geo-local-seo-generator'));
        
        // Contact display options
        $show_phone = get_option('seo_lead_pages_show_phone', 1);
        $show_email = get_option('seo_lead_pages_show_email', 1);
        $show_website = get_option('seo_lead_pages_show_website', 1);
        
        // Form styling options
        $form_background_color = get_option('seo_lead_pages_form_background_color', '#ffffff');
        $form_border_color = get_option('seo_lead_pages_form_border_color', '#e1e5e9');
        $form_text_color = get_option('seo_lead_pages_form_text_color', '#333333');
        $form_title_color = get_option('seo_lead_pages_form_title_color', '#0073aa');
        $input_background_color = get_option('seo_lead_pages_input_background_color', '#ffffff');
        $input_border_color = get_option('seo_lead_pages_input_border_color', '#ddd');
        $input_text_color = get_option('seo_lead_pages_input_text_color', '#333333');
        $button_background_color = get_option('seo_lead_pages_button_background_color', '#0073aa');
        $button_text_color = get_option('seo_lead_pages_button_text_color', '#ffffff');
        $button_hover_color = get_option('seo_lead_pages_button_hover_color', '#005a87');
        $form_border_radius = get_option('seo_lead_pages_form_border_radius', 8);
        $form_padding = get_option('seo_lead_pages_form_padding', 30);
        $form_width = get_option('seo_lead_pages_form_width', '100%');
        $form_alignment = get_option('seo_lead_pages_form_alignment', 'left');
        
        ?>
        <div class="wrap">
            <h1>📝 Form Settings</h1>
            <p>Customize the text labels and messages for your lead forms. These settings will be used for all generated pages and the shortcode.</p>
            
            <form method="post" action="">
                <?php wp_nonce_field('form_settings_action', 'form_settings_nonce'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Form Title</th>
                        <td>
                            <input type="text" name="form_title" value="<?php echo esc_attr($form_title); ?>" class="regular-text" />
                            <p class="description">The main title of the form (e.g., "Free SEO Advice for")</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Contact Label</th>
                        <td>
                            <input type="text" name="contact_label" value="<?php echo esc_attr($contact_label); ?>" class="regular-text" />
                            <p class="description">Label for the contact information section</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Name Field Label</th>
                        <td>
                            <input type="text" name="name_label" value="<?php echo esc_attr($name_label); ?>" class="regular-text" />
                            <p class="description">Label for the name input field</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Email Field Label</th>
                        <td>
                            <input type="text" name="email_label" value="<?php echo esc_attr($email_label); ?>" class="regular-text" />
                            <p class="description">Label for the email input field</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Phone Field Label</th>
                        <td>
                            <input type="text" name="phone_label" value="<?php echo esc_attr($phone_label); ?>" class="regular-text" />
                            <p class="description">Label for the phone input field</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Company Field Label</th>
                        <td>
                            <input type="text" name="company_label" value="<?php echo esc_attr($company_label); ?>" class="regular-text" />
                            <p class="description">Label for the company input field</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Message Field Label</th>
                        <td>
                            <input type="text" name="message_label" value="<?php echo esc_attr($message_label); ?>" class="regular-text" />
                            <p class="description">Label for the message textarea field</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Message Placeholder</th>
                        <td>
                            <textarea name="message_placeholder" rows="3" cols="50" class="large-text"><?php echo esc_textarea($message_placeholder); ?></textarea>
                            <p class="description">Placeholder text for the message textarea</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Submit Button Text</th>
                        <td>
                            <input type="text" name="submit_button" value="<?php echo esc_attr($submit_button); ?>" class="regular-text" />
                            <p class="description">Text for the submit button</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Contact Information Display</th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="checkbox" name="show_phone" value="1" <?php checked($show_phone, 1); ?> />
                                    Show phone number
                                </label><br>
                                <label>
                                    <input type="checkbox" name="show_email" value="1" <?php checked($show_email, 1); ?> />
                                    Show email address
                                </label><br>
                                <label>
                                    <input type="checkbox" name="show_website" value="1" <?php checked($show_website, 1); ?> />
                                    Show website URL
                                </label>
                            </fieldset>
                            <p class="description">Choose which contact information to display in the form</p>
                        </td>
                    </tr>
                </table>
                
                <h2 style="margin-top: 40px;">🎨 Form Styling Options</h2>
                <p>Customize the appearance of your lead forms on landing pages.</p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row" colspan="2"><h3 style="margin: 0;">Form Container</h3></th>
                    </tr>
                    
                    <tr>
                        <th scope="row">Background Color</th>
                        <td>
                            <input type="color" name="form_background_color" value="<?php echo esc_attr($form_background_color); ?>" />
                            <p class="description">Background color of the form container</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Border Color</th>
                        <td>
                            <input type="color" name="form_border_color" value="<?php echo esc_attr($form_border_color); ?>" />
                            <p class="description">Border color around the form</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Text Color</th>
                        <td>
                            <input type="color" name="form_text_color" value="<?php echo esc_attr($form_text_color); ?>" />
                            <p class="description">Default text color in the form</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Title Color</th>
                        <td>
                            <input type="color" name="form_title_color" value="<?php echo esc_attr($form_title_color); ?>" />
                            <p class="description">Color of the form title</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Border Radius</th>
                        <td>
                            <input type="number" name="form_border_radius" value="<?php echo esc_attr($form_border_radius); ?>" min="0" max="50" class="small-text" /> px
                            <p class="description">Rounded corners (0 = square, 50 = very rounded)</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Padding</th>
                        <td>
                            <input type="number" name="form_padding" value="<?php echo esc_attr($form_padding); ?>" min="0" max="100" class="small-text" /> px
                            <p class="description">Internal spacing inside the form</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Width</th>
                        <td>
                            <input type="text" name="form_width" value="<?php echo esc_attr($form_width); ?>" class="regular-text" placeholder="e.g., 100%, 500px, 80%" />
                            <p class="description">Form width (use % for responsive, px for fixed)</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Alignment</th>
                        <td>
                            <select name="form_alignment">
                                <option value="left" <?php selected($form_alignment, 'left'); ?>>Left</option>
                                <option value="center" <?php selected($form_alignment, 'center'); ?>>Center</option>
                                <option value="right" <?php selected($form_alignment, 'right'); ?>>Right</option>
                            </select>
                            <p class="description">Form alignment on the page</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row" colspan="2"><h3 style="margin: 20px 0 0;">Input Fields</h3></th>
                    </tr>
                    
                    <tr>
                        <th scope="row">Input Background</th>
                        <td>
                            <input type="color" name="input_background_color" value="<?php echo esc_attr($input_background_color); ?>" />
                            <p class="description">Background color of input fields</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Input Border</th>
                        <td>
                            <input type="color" name="input_border_color" value="<?php echo esc_attr($input_border_color); ?>" />
                            <p class="description">Border color of input fields</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Input Text</th>
                        <td>
                            <input type="color" name="input_text_color" value="<?php echo esc_attr($input_text_color); ?>" />
                            <p class="description">Text color in input fields</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row" colspan="2"><h3 style="margin: 20px 0 0;">Submit Button</h3></th>
                    </tr>
                    
                    <tr>
                        <th scope="row">Button Background</th>
                        <td>
                            <input type="color" name="button_background_color" value="<?php echo esc_attr($button_background_color); ?>" />
                            <p class="description">Background color of the submit button</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Button Text</th>
                        <td>
                            <input type="color" name="button_text_color" value="<?php echo esc_attr($button_text_color); ?>" />
                            <p class="description">Text color of the submit button</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Button Hover</th>
                        <td>
                            <input type="color" name="button_hover_color" value="<?php echo esc_attr($button_hover_color); ?>" />
                            <p class="description">Button color when hovering</p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button('Save Form Settings'); ?>
            </form>
            
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin-top: 30px;">
                <h3>🔍 Debug Information</h3>
                <p>Current styling values:</p>
                <ul>
                    <li><strong>Form Background:</strong> <?php echo esc_html($form_background_color); ?></li>
                    <li><strong>Form Border:</strong> <?php echo esc_html($form_border_color); ?></li>
                    <li><strong>Form Title:</strong> <?php echo esc_html($form_title_color); ?></li>
                    <li><strong>Button Background:</strong> <?php echo esc_html($button_background_color); ?></li>
                    <li><strong>Button Text:</strong> <?php echo esc_html($button_text_color); ?></li>
                    <li><strong>Button Hover:</strong> <?php echo esc_html($button_hover_color); ?></li>
                    <li><strong>Border Radius:</strong> <?php echo esc_html($form_border_radius); ?>px</li>
                    <li><strong>Padding:</strong> <?php echo esc_html($form_padding); ?>px</li>
                    <li><strong>Width:</strong> <?php echo esc_html($form_width); ?></li>
                    <li><strong>Alignment:</strong> <?php echo esc_html($form_alignment); ?></li>
                </ul>
            </div>
            
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin-top: 30px;">
                <h3>👀 Form Preview</h3>
                <p>Here's how your form will look with the current settings:</p>
                <?php echo do_shortcode('[seo_lead_form]'); ?>
            </div>
            
            <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px; margin-top: 30px;">
                <h3>📋 Shortcode Usage</h3>
                <p>You can use the lead form on any page or post with the following shortcode:</p>
                <code>[seo_lead_form]</code>
                
                <h4>Shortcode Options:</h4>
                <ul>
                    <li><code>[seo_lead_form title="Custom Title"]</code> - Custom form title</li>
                    <li><code>[seo_lead_form button_text="Custom Button"]</code> - Custom button text</li>
                    <li><code>[seo_lead_form title="Contact Us" button_text="Send Message"]</code> - Both options</li>
                </ul>
                
                <h4>Language Support:</h4>
                <p>The form text will automatically use your WordPress language settings. If you want to override specific texts, use the form settings above.</p>
            </div>
        </div>
        <?php
    }
    
    // Helper Functions
    private function generate_page_content($business_name, $location, $service, $phone, $email, $website) {
        $content = "<h1>{$service} in {$location} - {$business_name}</h1>\n\n";
        $content .= "<p>Welcome to {$business_name}, your trusted provider of {$service} in {$location}.</p>\n\n";
        $content .= "<h2>Our Services</h2>\n";
        $content .= "<p>We specialize in {$service} and are committed to providing the highest quality service to our customers in {$location}.</p>\n\n";
        $content .= "<h2>Contact Us</h2>\n";
        $content .= "<p>Ready to get started? Contact us today:</p>\n";
        $content .= "<ul>\n";
        if ($phone) {
            $content .= "<li>Phone: {$phone}</li>\n";
        }
        if ($email) {
            $content .= "<li>Email: {$email}</li>\n";
        }
        if ($website) {
            $content .= "<li>Website: <a href=\"{$website}\">{$website}</a></li>\n";
        }
        $content .= "</ul>\n\n";
        $content .= "<p>We look forward to serving you in {$location}!</p>";
        
        // Add lead form
        $content .= $this->generate_lead_form(array(
            'business_name' => $business_name,
            'city_name' => explode(',', $location)[0],
            'province_state' => explode(',', $location)[1] ?? '',
            'service' => $service,
            'phone' => $phone,
            'email' => $email,
            'website' => $website
        ));
        
        return $content;
    }
    
    private function generate_content_from_template($template_name, $location) {
        // Get saved templates
        $saved_templates = get_option('seo_lead_pages_templates', array());
        
        error_log('GEO Local SEO: Looking for template: ' . $template_name);
        error_log('GEO Local SEO: Available templates: ' . print_r(array_keys($saved_templates), true));
        error_log('GEO Local SEO: Location data: ' . print_r($location, true));
        
        // Find the template
        $template_content = '';
        if (isset($saved_templates[$template_name])) {
            $template_content = $saved_templates[$template_name];
        error_log('GEO Local SEO: Template found! Content length: ' . strlen($template_content));
        error_log('GEO Local SEO: Template content preview: ' . substr($template_content, 0, 200) . '...');
        
        // Debug: Check if template contains service_name shortcode
        if (strpos($template_content, '[service_name]') !== false) {
            error_log('GEO Local SEO: Template contains [service_name] shortcode');
        } else {
            error_log('GEO Local SEO: WARNING - Template does NOT contain [service_name] shortcode!');
        }
        }
        
        if (empty($template_content)) {
            error_log('GEO Local SEO: Template not found: ' . $template_name);
            error_log('GEO Local SEO: Falling back to default content');
            // Fallback to default content
            return $this->generate_page_content($location['business_name'], $location['city_name'] . ', ' . $location['province_state'], $location['service'], $location['phone'], $location['email'], $location['website']);
        }
        
        // Replace shortcodes
        error_log('GEO Local SEO: Replacing shortcodes...');
        error_log('GEO Local SEO: Location data for shortcode replacement: ' . print_r($location, true));
        $content = $this->replace_shortcodes($template_content, $location);
        error_log('GEO Local SEO: After shortcode replacement: ' . substr($content, 0, 200) . '...');
        
        // Process spintax
        error_log('GEO Local SEO: Processing spintax...');
        error_log('GEO Local SEO: Content before spintax: ' . substr($content, 0, 300) . '...');
        $content = $this->process_spintax($content);
        error_log('GEO Local SEO: After spintax processing: ' . substr($content, 0, 300) . '...');
        
        // Process WordPress shortcodes
        error_log('GEO Local SEO: Processing WordPress shortcodes...');
        $content = do_shortcode($content);
        error_log('GEO Local SEO: After WordPress shortcodes: ' . substr($content, 0, 200) . '...');
        
        // Add lead form to content
        error_log('GEO Local SEO: Adding lead form to content...');
        $lead_form = $this->generate_lead_form($location);
        $content .= $lead_form;
        
        error_log('GEO Local SEO: Final content length: ' . strlen($content));
        error_log('GEO Local SEO: Final content preview: ' . substr($content, 0, 500) . '...');
        return $content;
    }
    
    private function replace_shortcodes($content, $location) {
        // Get custom shortcode mappings
        $shortcode_mappings = get_option('seo_lead_pages_shortcode_mappings', array());
        
        // Define shortcode replacements with fallbacks
        // NEW PRIORITY: 1. Shortcode settings (user choice), 2. Location data, 3. Defaults
        $replacements = array(
            '[service_name]' => (isset($shortcode_mappings['service_name']) && !empty($shortcode_mappings['service_name'])) ? $shortcode_mappings['service_name'] : (!empty($location['service']) ? $location['service'] : 'SEO Diensten'),
            '[location_name]' => !empty($location['city_name']) ? $location['city_name'] : (isset($shortcode_mappings['location_name']) ? $shortcode_mappings['location_name'] : 'Amsterdam'),
            '[province_name]' => !empty($location['province_state']) ? $location['province_state'] : (isset($shortcode_mappings['province_name']) ? $shortcode_mappings['province_name'] : 'Noord-Holland'),
            '[business_name]' => (isset($shortcode_mappings['business_name']) && !empty($shortcode_mappings['business_name'])) ? $shortcode_mappings['business_name'] : (!empty($location['business_name']) ? $location['business_name'] : get_option('blogname', 'Uw Bedrijf')),
            '[phone_number]' => (isset($shortcode_mappings['phone_number']) && !empty($shortcode_mappings['phone_number'])) ? $shortcode_mappings['phone_number'] : (!empty($location['phone']) ? $location['phone'] : '+31 20 123 4567'),
            '[contact_email]' => (isset($shortcode_mappings['contact_email']) && !empty($shortcode_mappings['contact_email'])) ? $shortcode_mappings['contact_email'] : (!empty($location['email']) ? $location['email'] : get_option('admin_email')),
            '[website_url]' => !empty($location['website']) ? $location['website'] : (isset($shortcode_mappings['website_url']) ? $shortcode_mappings['website_url'] : get_option('home')),
            '[contact_form]' => 'contact formulier', // Nederlandse placeholder
            '[seo_lead_form]' => '[seo_lead_form]', // Keep as shortcode for WordPress processing
        );
        
        // Debug: Log the decision making process
        foreach ($replacements as $shortcode => $replacement) {
            $shortcode_key = str_replace(['[', ']'], '', $shortcode);
            $location_value = '';
            $settings_value = '';
            
            switch ($shortcode_key) {
                case 'service_name':
                    $location_value = $location['service'] ?? '';
                    $settings_value = $shortcode_mappings['service_name'] ?? '';
                    break;
                case 'business_name':
                    $location_value = $location['business_name'] ?? '';
                    $settings_value = $shortcode_mappings['business_name'] ?? '';
                    break;
                case 'phone_number':
                    $location_value = $location['phone'] ?? '';
                    $settings_value = $shortcode_mappings['phone_number'] ?? '';
                    break;
                case 'contact_email':
                    $location_value = $location['email'] ?? '';
                    $settings_value = $shortcode_mappings['contact_email'] ?? '';
                    break;
            }
            
            if (in_array($shortcode_key, ['service_name', 'business_name', 'phone_number', 'contact_email'])) {
                error_log("GEO Local SEO: {$shortcode} - Settings: '{$settings_value}' | Location: '{$location_value}' | Final: '{$replacement}' (NEW PRIORITY: Settings first!)");
            }
        }
        
        // Debug: Log what we're using for replacements
        error_log('GEO Local SEO: Shortcode mappings from database: ' . print_r($shortcode_mappings, true));
        error_log('GEO Local SEO: Final replacements being used: ' . print_r($replacements, true));
        
        // Extra debug for service_name specifically
        error_log('GEO Local SEO: SERVICE_NAME DEBUG - Settings: "' . (isset($shortcode_mappings['service_name']) ? $shortcode_mappings['service_name'] : 'NOT SET') . '" | Location: "' . (isset($location['service']) ? $location['service'] : 'NOT SET') . '" | Final: "' . $replacements['[service_name]'] . '"');
        
        // Replace shortcodes
        foreach ($replacements as $shortcode => $replacement) {
            $old_content = $content;
            $content = str_replace($shortcode, $replacement, $content);
            if ($old_content !== $content) {
                error_log('GEO Local SEO: Replaced ' . $shortcode . ' with: ' . $replacement);
            } else {
                // Debug: Check if shortcode exists in content
                if (strpos($content, $shortcode) !== false) {
                    error_log('GEO Local SEO: WARNING - ' . $shortcode . ' found in content but not replaced!');
                }
            }
        }
        
        error_log('GEO Local SEO: Shortcode replacement completed');
        
        // Debug: Check final content for service_name
        if (strpos($content, '[service_name]') !== false) {
            error_log('GEO Local SEO: WARNING - [service_name] still found in final content!');
        } else {
            error_log('GEO Local SEO: [service_name] successfully replaced in final content');
        }
        
        // Debug: Show final content preview
        error_log('GEO Local SEO: Final content preview after shortcode replacement: ' . substr($content, 0, 300) . '...');
        
        return $content;
    }
    
    private function generate_page_title_and_slug($service, $location, $url_logic = 'service_in_location', $custom_slug_template = '') {
        // Get custom shortcode mappings to use for title generation
        $shortcode_mappings = get_option('seo_lead_pages_shortcode_mappings', array());
        
        // Use shortcode settings for service name if available, otherwise use provided service
        $service_for_title = (!empty($shortcode_mappings['service_name'])) ? $shortcode_mappings['service_name'] : $service;
        
        // Use slug template from shortcode settings if available, but allow URL logic override
        // If url_logic is provided and different from shortcode settings, use url_logic as override
        if (!empty($url_logic) && $url_logic !== $shortcode_mappings['slug_template']) {
            $slug_template = $url_logic;
            error_log('GEO Local SEO: Using URL logic override: "' . $url_logic . '" instead of shortcode setting: "' . $shortcode_mappings['slug_template'] . '"');
        } else {
            $slug_template = (isset($shortcode_mappings['slug_template']) && !empty($shortcode_mappings['slug_template'])) ? $shortcode_mappings['slug_template'] : $url_logic;
        }
        
        error_log('GEO Local SEO: Title generation - Original service: "' . $service . '" | Shortcode service: "' . $shortcode_mappings['service_name'] . '" | Final service: "' . $service_for_title . '" | Slug template: "' . $slug_template . '"');
        error_log('GEO Local SEO: Full shortcode mappings: ' . print_r($shortcode_mappings, true));
        
        $city_name = $location['city_name'];
        $province_state = $location['province_state'];
        $country = $location['country'];
        
        // Get localized names
        $localized = $this->get_localized_names($country, $province_state);
        $localized_province = $localized['province'];
        $localized_country = $localized['country'];
        
        // Generate title and slug based on slug template from shortcode settings or page generator
        if ($slug_template === 'custom' && (!empty($custom_slug_template) || !empty($shortcode_mappings['custom_slug_template']))) {
            // Use custom slug template from page generator or shortcode settings
            $custom_template = !empty($custom_slug_template) ? $custom_slug_template : $shortcode_mappings['custom_slug_template'];
            
            // Replace variables in custom template
            $slug = $custom_template;
            $slug = str_replace('[service_name]', sanitize_title($service_for_title), $slug);
            $slug = str_replace('[location_name]', sanitize_title($city_name), $slug);
            $slug = str_replace('[province_name]', sanitize_title($localized_province), $slug);
            $slug = str_replace('[business_name]', sanitize_title($shortcode_mappings['business_name'] ?? ''), $slug);
            $slug = str_replace('[country]', sanitize_title($localized_country), $slug);
            
            // Generate title from slug (capitalize first letters)
            $title = ucwords(str_replace('-', ' ', $slug));
            
            error_log('GEO Local SEO: Using custom slug template: "' . $custom_template . '" → "' . $slug . '"');
        } else {
            // Use predefined templates
            switch ($slug_template) {
                case 'service_location':
                    $title = "{$service_for_title} {$city_name}";
                    $slug = sanitize_title("{$service_for_title} {$city_name}");
                    break;
                    
                case 'service_in_location':
                    $title = "{$service_for_title} in {$city_name}";
                    $slug = sanitize_title("{$service_for_title} in {$city_name}");
                    break;
                    
                case 'location_service':
                    $title = "{$city_name} {$service_for_title}";
                    $slug = sanitize_title("{$city_name} {$service_for_title}");
                    break;
                    
                case 'service_in_location_province':
                    $title = "{$service_for_title} in {$city_name}, {$localized_province}";
                    $slug = sanitize_title("{$service_for_title} in {$city_name} {$localized_province}");
                    break;
                    
                case 'service_in_location_country':
                    $title = "{$service_for_title} in {$city_name}, {$localized_country}";
                    $slug = sanitize_title("{$service_for_title} in {$city_name} {$localized_country}");
                    break;
                    
                case 'service_in_location_province_country':
                    $title = "{$service_for_title} in {$city_name}, {$localized_province}, {$localized_country}";
                    $slug = sanitize_title("{$service_for_title} in {$city_name} {$localized_province} {$localized_country}");
                    break;
                    
                default:
                    $title = "{$service_for_title} in {$city_name}";
                    $slug = sanitize_title("{$service_for_title} in {$city_name}");
                    break;
            }
        }
        
        return array(
            'title' => $title,
            'slug' => $slug
        );
    }
    
    private function get_localized_names($country, $province = '') {
        // Mapping for localized country and province names
        $localized_mapping = array(
            'Netherlands' => array(
                'country' => 'Nederland',
                'provinces' => array(
                    'North Holland' => 'Noord-Holland',
                    'South Holland' => 'Zuid-Holland',
                    'Utrecht' => 'Utrecht',
                    'Gelderland' => 'Gelderland',
                    'Overijssel' => 'Overijssel',
                    'Drenthe' => 'Drenthe',
                    'Friesland' => 'Friesland',
                    'Groningen' => 'Groningen',
                    'Flevoland' => 'Flevoland',
                    'North Brabant' => 'Noord-Brabant',
                    'Limburg' => 'Limburg',
                    'Zeeland' => 'Zeeland'
                )
            ),
            'Belgium' => array(
                'country' => 'België',
                'provinces' => array(
                    'Antwerp' => 'Antwerpen',
                    'East Flanders' => 'Oost-Vlaanderen',
                    'Flemish Brabant' => 'Vlaams-Brabant',
                    'West Flanders' => 'West-Vlaanderen',
                    'Hainaut' => 'Henegouwen',
                    'Liège' => 'Luik',
                    'Limburg' => 'Limburg',
                    'Luxembourg' => 'Luxemburg',
                    'Namur' => 'Namen',
                    'Walloon Brabant' => 'Waals-Brabant'
                )
            ),
            'Germany' => array(
                'country' => 'Deutschland',
                'provinces' => array(
                    'Bavaria' => 'Bayern',
                    'Baden-Württemberg' => 'Baden-Württemberg',
                    'North Rhine-Westphalia' => 'Nordrhein-Westfalen',
                    'Lower Saxony' => 'Niedersachsen',
                    'Hesse' => 'Hessen',
                    'Saxony' => 'Sachsen',
                    'Rhineland-Palatinate' => 'Rheinland-Pfalz',
                    'Brandenburg' => 'Brandenburg',
                    'Thuringia' => 'Thüringen',
                    'Saxony-Anhalt' => 'Sachsen-Anhalt',
                    'Mecklenburg-Vorpommern' => 'Mecklenburg-Vorpommern',
                    'Schleswig-Holstein' => 'Schleswig-Holstein',
                    'Saarland' => 'Saarland',
                    'Bremen' => 'Bremen',
                    'Hamburg' => 'Hamburg',
                    'Berlin' => 'Berlin'
                )
            ),
            'France' => array(
                'country' => 'France',
                'provinces' => array(
                    'Île-de-France' => 'Île-de-France',
                    'Auvergne-Rhône-Alpes' => 'Auvergne-Rhône-Alpes',
                    'Occitanie' => 'Occitanie',
                    'Nouvelle-Aquitaine' => 'Nouvelle-Aquitaine',
                    'Grand Est' => 'Grand Est',
                    'Hauts-de-France' => 'Hauts-de-France',
                    'Provence-Alpes-Côte d\'Azur' => 'Provence-Alpes-Côte d\'Azur',
                    'Brittany' => 'Bretagne',
                    'Pays de la Loire' => 'Pays de la Loire',
                    'Normandy' => 'Normandie',
                    'Bourgogne-Franche-Comté' => 'Bourgogne-Franche-Comté',
                    'Centre-Val de Loire' => 'Centre-Val de Loire',
                    'Corsica' => 'Corse'
                )
            ),
            'Spain' => array(
                'country' => 'España',
                'provinces' => array(
                    'Andalusia' => 'Andalucía',
                    'Catalonia' => 'Cataluña',
                    'Madrid' => 'Madrid',
                    'Valencia' => 'Comunidad Valenciana',
                    'Galicia' => 'Galicia',
                    'Castile and León' => 'Castilla y León',
                    'Basque Country' => 'País Vasco',
                    'Canary Islands' => 'Islas Canarias',
                    'Castilla-La Mancha' => 'Castilla-La Mancha',
                    'Murcia' => 'Región de Murcia',
                    'Aragon' => 'Aragón',
                    'Extremadura' => 'Extremadura',
                    'Balearic Islands' => 'Islas Baleares',
                    'Asturias' => 'Principado de Asturias',
                    'Navarre' => 'Comunidad Foral de Navarra',
                    'Cantabria' => 'Cantabria',
                    'La Rioja' => 'La Rioja'
                )
            ),
            'Italy' => array(
                'country' => 'Italia',
                'provinces' => array(
                    'Lombardy' => 'Lombardia',
                    'Lazio' => 'Lazio',
                    'Campania' => 'Campania',
                    'Sicily' => 'Sicilia',
                    'Veneto' => 'Veneto',
                    'Emilia-Romagna' => 'Emilia-Romagna',
                    'Piedmont' => 'Piemonte',
                    'Puglia' => 'Puglia',
                    'Tuscany' => 'Toscana',
                    'Calabria' => 'Calabria',
                    'Sardinia' => 'Sardegna',
                    'Liguria' => 'Liguria',
                    'Marche' => 'Marche',
                    'Abruzzo' => 'Abruzzo',
                    'Friuli-Venezia Giulia' => 'Friuli-Venezia Giulia',
                    'Trentino-Alto Adige' => 'Trentino-Alto Adige',
                    'Umbria' => 'Umbria',
                    'Basilicata' => 'Basilicata',
                    'Molise' => 'Molise',
                    'Aosta Valley' => 'Valle d\'Aosta'
                )
            ),
            'United Kingdom' => array(
                'country' => 'United Kingdom',
                'provinces' => array(
                    'England' => 'England',
                    'Scotland' => 'Scotland',
                    'Wales' => 'Wales',
                    'Northern Ireland' => 'Northern Ireland'
                )
            ),
            'USA' => array(
                'country' => 'United States',
                'provinces' => array(
                    'California' => 'California',
                    'Texas' => 'Texas',
                    'Florida' => 'Florida',
                    'New York' => 'New York',
                    'Pennsylvania' => 'Pennsylvania',
                    'Illinois' => 'Illinois',
                    'Ohio' => 'Ohio',
                    'Georgia' => 'Georgia',
                    'North Carolina' => 'North Carolina',
                    'Michigan' => 'Michigan'
                )
            )
        );
        
        $result = array();
        
        // Get localized country name
        if (isset($localized_mapping[$country])) {
            $result['country'] = $localized_mapping[$country]['country'];
        } else {
            $result['country'] = $country; // Fallback to original name
        }
        
        // Get localized province name
        if (!empty($province) && isset($localized_mapping[$country]['provinces'][$province])) {
            $result['province'] = $localized_mapping[$country]['provinces'][$province];
        } else {
            $result['province'] = $province; // Fallback to original name
        }
        
        return $result;
    }
    
    private function generate_lead_form($location) {
        // Get form text settings
        $form_title = get_option('seo_lead_pages_form_title', __('Free SEO Advice for', 'geo-local-seo-generator'));
        $contact_label = get_option('seo_lead_pages_contact_label', __('Direct contact:', 'geo-local-seo-generator'));
        $name_label = get_option('seo_lead_pages_name_label', __('Name', 'geo-local-seo-generator'));
        $email_label = get_option('seo_lead_pages_email_label', __('Email address', 'geo-local-seo-generator'));
        $phone_label = get_option('seo_lead_pages_phone_label', __('Phone number', 'geo-local-seo-generator'));
        $company_label = get_option('seo_lead_pages_company_label', __('Company name', 'geo-local-seo-generator'));
        $message_label = get_option('seo_lead_pages_message_label', __('Message', 'geo-local-seo-generator'));
        $message_placeholder = get_option('seo_lead_pages_message_placeholder', __('Tell us about your SEO goals and challenges...', 'geo-local-seo-generator'));
        $submit_button = get_option('seo_lead_pages_submit_button', __('Request Free SEO Advice', 'geo-local-seo-generator'));
        
        // Get contact display options
        $show_phone = get_option('seo_lead_pages_show_phone', 1);
        $show_email = get_option('seo_lead_pages_show_email', 1);
        $show_website = get_option('seo_lead_pages_show_website', 1);
        
        // Get form styling options
        $form_background_color = get_option('seo_lead_pages_form_background_color', '#ffffff');
        $form_border_color = get_option('seo_lead_pages_form_border_color', '#e1e5e9');
        $form_text_color = get_option('seo_lead_pages_form_text_color', '#333333');
        $form_title_color = get_option('seo_lead_pages_form_title_color', '#0073aa');
        $input_background_color = get_option('seo_lead_pages_input_background_color', '#ffffff');
        $input_border_color = get_option('seo_lead_pages_input_border_color', '#ddd');
        $input_text_color = get_option('seo_lead_pages_input_text_color', '#333333');
        $button_background_color = get_option('seo_lead_pages_button_background_color', '#0073aa');
        $button_text_color = get_option('seo_lead_pages_button_text_color', '#ffffff');
        $button_hover_color = get_option('seo_lead_pages_button_hover_color', '#005a87');
        $form_border_radius = get_option('seo_lead_pages_form_border_radius', 8);
        $form_padding = get_option('seo_lead_pages_form_padding', 30);
        $form_width = get_option('seo_lead_pages_form_width', '100%');
        $form_alignment = get_option('seo_lead_pages_form_alignment', 'left');
        
        // Debug: Log the styling options
        error_log('GEO Local SEO Debug - Generate form styling options:');
        error_log('Background: ' . $form_background_color);
        error_log('Button: ' . $button_background_color);
        error_log('Title: ' . $form_title_color);
        
        // Build contact info section
        $contact_info = '';
        if ($show_phone || $show_email || $show_website) {
            $contact_info = '<div class="contact-info">';
            $contact_info .= '<p><strong>' . esc_html($contact_label) . '</strong></p>';
            
            if ($show_phone && !empty($location['phone'])) {
                $contact_info .= '<p>📞 <a href="tel:' . esc_attr($location['phone']) . '">' . esc_html($location['phone']) . '</a></p>';
            }
            
            if ($show_email && !empty($location['email'])) {
                $contact_info .= '<p>📧 <a href="mailto:' . esc_attr($location['email']) . '">' . esc_html($location['email']) . '</a></p>';
            }
            
            if ($show_website && !empty($location['website'])) {
                $contact_info .= '<p>🌐 <a href="' . esc_url($location['website']) . '" target="_blank">' . esc_html($location['website']) . '</a></p>';
            }
            
            $contact_info .= '</div>';
        }
        
        $form_html = '
        <style>
        .seo-lead-form {
            background: ' . esc_attr($form_background_color) . ' !important;
            border: 1px solid ' . esc_attr($form_border_color) . ' !important;
            color: ' . esc_attr($form_text_color) . ' !important;
            border-radius: ' . esc_attr($form_border_radius) . 'px !important;
            padding: ' . esc_attr($form_padding) . 'px !important;
            width: ' . esc_attr($form_width) . ' !important;
            text-align: ' . esc_attr($form_alignment) . ' !important;
        }
        
        .seo-lead-form h3 {
            color: ' . esc_attr($form_title_color) . ' !important;
        }
        
        .seo-lead-form .form-row input,
        .seo-lead-form .form-row textarea {
            background: ' . esc_attr($input_background_color) . ' !important;
            border: 2px solid ' . esc_attr($input_border_color) . ' !important;
            color: ' . esc_attr($input_text_color) . ' !important;
        }
        
        .seo-lead-form .submit-button {
            background: ' . esc_attr($button_background_color) . ' !important;
            color: ' . esc_attr($button_text_color) . ' !important;
        }
        
        .seo-lead-form .submit-button:hover {
            background: ' . esc_attr($button_hover_color) . ' !important;
        }
        </style>
        
        <div class="seo-lead-form">
            <h3>🎯 ' . esc_html($form_title) . ' ' . esc_html($location['city_name']) . '</h3>
            
            ' . $contact_info . '
            
            <form id="lead-form-' . uniqid() . '" class="lead-form">
                <div class="form-row">
                    <label for="name">' . esc_html($name_label) . ' *</label>
                    <input type="text" id="name" name="name" required>
                </div>
                
                <div class="form-row">
                    <label for="email">' . esc_html($email_label) . ' *</label>
                    <input type="email" id="email" name="email" required>
                </div>
                
                <div class="form-row">
                    <label for="phone">' . esc_html($phone_label) . '</label>
                    <input type="tel" id="phone" name="phone">
                </div>
                
                <div class="form-row">
                    <label for="company">' . esc_html($company_label) . '</label>
                    <input type="text" id="company" name="company">
                </div>
                
                <div class="form-row">
                    <label for="message">' . esc_html($message_label) . '</label>
                    <textarea id="message" name="message" placeholder="' . esc_attr($message_placeholder) . '"></textarea>
                </div>
                
                <button type="submit" class="submit-button">🚀 ' . esc_html($submit_button) . '</button>
                
                <input type="hidden" name="service" value="' . esc_attr($location['service']) . '">
                <input type="hidden" name="location" value="' . esc_attr($location['city_name'] . ', ' . $location['province_state']) . '">
                <input type="hidden" name="action" value="submit_lokaal_lead_fixed">
                <input type="hidden" name="nonce" value="' . wp_create_nonce('seo_lead_pages_nonce') . '">
                
                <div class="form-message" style="display: none;"></div>
            </form>
        </div>';
        
        return $form_html;
    }
    
    private function process_spintax($content) {
        // Process spintax syntax {option1|option2|option3} including nested spintax
        // Keep processing until no more spintax patterns are found
        $max_iterations = 20; // Increased iterations for complex spintax
        $iteration = 0;
        
        error_log('GEO Local SEO: Starting spintax processing...');
        
        while ($iteration < $max_iterations) {
            // More robust pattern that handles nested spintax better
            $pattern = '/\{([^{}]*(?:\{[^{}]*\}[^{}]*)*)\}/';
            $new_content = preg_replace_callback($pattern, function($matches) {
                $options = explode('|', $matches[1]);
                $selected_option = $options[array_rand($options)];
                
                error_log('GEO Local SEO: Spintax processing - Original: "' . $matches[0] . '" | Selected: "' . $selected_option . '"');
                
                // Process any nested spintax in the selected option
                return $this->process_spintax_recursive($selected_option);
            }, $content);
            
            // If no changes were made, we're done
            if ($new_content === $content) {
                error_log('GEO Local SEO: No more spintax patterns found, stopping at iteration ' . $iteration);
                break;
            }
            
            $content = $new_content;
            $iteration++;
        }
        
        error_log('GEO Local SEO: Spintax processing completed after ' . $iteration . ' iterations');
        return $content;
    }
    
    private function process_spintax_recursive($content) {
        // Process nested spintax patterns
        $pattern = '/\{([^{}]*(?:\{[^{}]*\}[^{}]*)*)\}/';
        
        return preg_replace_callback($pattern, function($matches) {
            $options = explode('|', $matches[1]);
            return $options[array_rand($options)];
        }, $content);
    }
    
    public function ajax_submit_lead() {
        // Check nonce for security
        if (!wp_verify_nonce($_POST['nonce'], 'seo_lead_pages_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        // Sanitize input data
        $name = sanitize_text_field($_POST['name']);
        $email = sanitize_email($_POST['email']);
        $phone = sanitize_text_field($_POST['phone'] ?? '');
        $company = sanitize_text_field($_POST['company'] ?? '');
        $message = sanitize_textarea_field($_POST['message']);
        $service = sanitize_text_field($_POST['service']);
        $location = sanitize_text_field($_POST['location']);
        $page_id = isset($_POST['page_id']) ? intval($_POST['page_id']) : null;
        
        // Validate required fields
        if (empty($name) || empty($email)) {
            wp_send_json_error('Name and email are required');
        }
        
        // Validate email format
        if (!is_email($email)) {
            wp_send_json_error('Invalid email format');
        }
        
        // Save lead to database
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_lead_pages_leads';
        
        $result = $wpdb->insert(
            $table_name,
            array(
                'name' => $name,
                'email' => $email,
                'phone' => $phone,
                'company' => $company,
                'message' => $message,
                'service' => $service,
                'location' => $location,
                'page_id' => $page_id,
                'status' => 'new',
                'created_at' => current_time('mysql')
            ),
            array(
                '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s'
            )
        );
        
        if ($result === false) {
            wp_send_json_error('Failed to save lead');
        }
        
        // Send email notification (optional)
        $this->send_lead_notification($name, $email, $phone, $company, $message, $service, $location);
        
        wp_send_json_success('Lead saved successfully');
    }
    
    private function send_lead_notification($name, $email, $phone, $company, $message, $service, $location) {
        $admin_email = get_option('admin_email');
        $subject = 'New Lead: ' . $service . ' in ' . $location;
        
        $body = "New lead received:\n\n";
        $body .= "Name: " . $name . "\n";
        $body .= "Email: " . $email . "\n";
        $body .= "Phone: " . $phone . "\n";
        $body .= "Company: " . $company . "\n";
        $body .= "Service: " . $service . "\n";
        $body .= "Location: " . $location . "\n";
        $body .= "Message: " . $message . "\n";
        $body .= "Date: " . current_time('Y-m-d H:i:s') . "\n";
        
        wp_mail($admin_email, $subject, $body);
        
        wp_send_json_success('Lead submitted successfully!');
    }
    
    public function ajax_get_lead_stats() {
        if (!wp_verify_nonce($_POST['nonce'], 'seo_lead_pages_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        global $wpdb;
        
        $total = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}seo_lead_pages_leads");
        $open = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}seo_lead_pages_leads WHERE status = 'open'");
        $in_progress = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}seo_lead_pages_leads WHERE status = 'in_progress'");
        $completed = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}seo_lead_pages_leads WHERE status = 'completed'");
        
        wp_send_json_success(array(
            'total' => $total,
            'open' => $open,
            'in_progress' => $in_progress,
            'completed' => $completed
        ));
    }
    
    public function ajax_get_leads_table() {
        if (!wp_verify_nonce($_POST['nonce'], 'seo_lead_pages_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        global $wpdb;
        $leads = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}seo_lead_pages_leads ORDER BY created_at DESC LIMIT 50");
        
        ob_start();
        ?>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th><input type="checkbox" id="select-all-leads"></th>
                    <th>Name</th>
                    <th>Email</th>
                    <th>Phone</th>
                    <th>Company</th>
                    <th>Service</th>
                    <th>Location</th>
                    <th>Status</th>
                    <th>Date</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (!empty($leads)): ?>
                    <?php foreach ($leads as $lead): ?>
                        <tr>
                            <td><input type="checkbox" class="lead-checkbox" value="<?php echo $lead->id; ?>"></td>
                            <td><?php echo esc_html($lead->name); ?></td>
                            <td><?php echo esc_html($lead->email); ?></td>
                            <td><?php echo esc_html($lead->phone); ?></td>
                            <td><?php echo esc_html($lead->company); ?></td>
                            <td><?php echo esc_html($lead->service); ?></td>
                            <td><?php echo esc_html($lead->location); ?></td>
                            <td>
                                <select class="lead-status-select" data-lead-id="<?php echo $lead->id; ?>">
                                    <option value="open" <?php selected($lead->status, 'open'); ?>>Open</option>
                                    <option value="in_progress" <?php selected($lead->status, 'in_progress'); ?>>In Progress</option>
                                    <option value="rejected" <?php selected($lead->status, 'rejected'); ?>>Rejected</option>
                                    <option value="completed" <?php selected($lead->status, 'completed'); ?>>Completed</option>
                                </select>
                            </td>
                            <td><?php echo date('Y-m-d H:i', strtotime($lead->created_at)); ?></td>
                            <td>
                                <button class="button button-small delete-lead" data-lead-id="<?php echo $lead->id; ?>">Delete</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="10" style="text-align: center; padding: 20px;">No leads found.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
        
        <script>
        jQuery(document).ready(function($) {
            $('#select-all-leads').on('change', function() {
                $('.lead-checkbox').prop('checked', $(this).is(':checked'));
            });
        });
        </script>
        <?php
        
        wp_send_json_success(ob_get_clean());
    }
    
    public function ajax_update_lead_status() {
        if (!wp_verify_nonce($_POST['nonce'], 'seo_lead_pages_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        global $wpdb;
        
        $lead_id = intval($_POST['lead_id']);
        $status = sanitize_text_field($_POST['status']);
        
        $result = $wpdb->update(
            $wpdb->prefix . 'seo_lead_pages_leads',
            array('status' => $status),
            array('id' => $lead_id),
            array('%s'),
            array('%d')
        );
        
        if ($result !== false) {
            wp_send_json_success('Lead status updated successfully!');
        } else {
            wp_send_json_error('Failed to update lead status.');
        }
    }
    
    public function ajax_delete_lead() {
        if (!wp_verify_nonce($_POST['nonce'], 'seo_lead_pages_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        global $wpdb;
        
        $lead_id = intval($_POST['lead_id']);
        
        $result = $wpdb->delete(
            $wpdb->prefix . 'seo_lead_pages_leads',
            array('id' => $lead_id),
            array('%d')
        );
        
        if ($result !== false) {
            wp_send_json_success('Lead deleted successfully!');
        } else {
            wp_send_json_error('Failed to delete lead.');
        }
    }
    
    public function ajax_bulk_delete_leads() {
        if (!wp_verify_nonce($_POST['nonce'], 'seo_lead_pages_nonce')) {
            wp_send_json_error('Security check failed');
        }
        
        global $wpdb;
        
        $lead_ids = array_map('intval', $_POST['lead_ids']);
        $placeholders = implode(',', array_fill(0, count($lead_ids), '%d'));
        
        $result = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$wpdb->prefix}seo_lead_pages_leads WHERE id IN ($placeholders)",
            $lead_ids
        ));
        
        if ($result !== false) {
            wp_send_json_success('Leads deleted successfully!');
        } else {
            wp_send_json_error('Failed to delete leads.');
        }
    }
    
    public function ajax_export_leads_csv() {
        if (!wp_verify_nonce($_GET['nonce'], 'seo_lead_pages_nonce')) {
            wp_die('Security check failed');
        }
        
        global $wpdb;
        $leads = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}seo_lead_pages_leads ORDER BY created_at DESC");
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="leads-export-' . date('Y-m-d-H-i-s') . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // CSV headers
        fputcsv($output, array('ID', 'Name', 'Email', 'Phone', 'Company', 'Message', 'Service', 'Location', 'Status', 'Created At', 'Page URL'));
        
        foreach ($leads as $lead) {
            fputcsv($output, array(
                $lead->id,
                $lead->name,
                $lead->email,
                $lead->phone,
                $lead->company,
                $lead->message,
                $lead->service,
                $lead->location,
                $lead->status,
                $lead->created_at,
                get_permalink($lead->page_id)
            ));
        }
        
        fclose($output);
        exit;
    }
}
