<?php
/**
 * Plugin Name: GEO - Local SEO - Lead Pages Generator
 * Plugin URI: https://stormseo.nl/geo-local-seo-plugin/
 * Description: Generate SEO-optimized landing pages for local services and locations automatically. Includes Template Maker, Locations Manager, AI Content, A/B Testing, Advanced Analytics, and comprehensive local SEO features.
 * Version: 1.0.1
 * Author: Storm SEO
 * Author URI: https://stormseo.nl
 * License: GPL v2 or later
 * Text Domain: geo-local-seo-generator
 * Domain Path: /languages
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('SEO_LEAD_PAGES_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SEO_LEAD_PAGES_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SEO_LEAD_PAGES_PLUGIN_VERSION', '1.0.1');

// Include required files (guarded) - support both 'includes' and 'Includes' directories
$__seo_lead_pages_required = array(
    'class-admin.php',
    'class-frontend.php',
    'class-database.php',
    'class-shortcodes.php',
);

foreach ($__seo_lead_pages_required as $file) {
    $file_path = SEO_LEAD_PAGES_PLUGIN_DIR . 'includes/' . $file;
    if (file_exists($file_path)) {
        require_once $file_path;
    }
}

// Main plugin class
class GEO_Local_SEO_Lead_Pages_Generator {
    
    private $admin;
    private $frontend;
    private $database;
    private $design_manager;
    private $seo_manager;
    private $gmb;
    private $analytics;
    private $template_manager;
    private $ai_content_manager;
    private $pagespeed_manager;
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('plugins_loaded', array($this, 'plugins_loaded'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    public function init() {
        // Initialize admin
        if (class_exists('GEO_Local_SEO_Admin')) {
            $this->admin = new GEO_Local_SEO_Admin();
        }
        
        // Initialize frontend
        if (class_exists('SEO_Lead_Pages_Frontend')) {
            $this->frontend = new SEO_Lead_Pages_Frontend();
        }
        
        // Initialize shortcodes
        if (class_exists('SEO_Lead_Pages_Shortcodes')) {
            new SEO_Lead_Pages_Shortcodes();
        }
        
        // Initialize other components if they exist
        if (class_exists('SEO_Lead_Pages_Database')) {
            $this->database = new SEO_Lead_Pages_Database();
        }
        
        if (class_exists('SEO_Lead_Pages_Design_Manager')) {
            $this->design_manager = new SEO_Lead_Pages_Design_Manager();
        }
        
        if (class_exists('SEO_Lead_Pages_SEO_Manager')) {
            $this->seo_manager = new SEO_Lead_Pages_SEO_Manager();
        }
        
        if (class_exists('SEO_Lead_Pages_Google_My_Business')) {
            $this->gmb = new SEO_Lead_Pages_Google_My_Business();
        }
        
        if (class_exists('SEO_Lead_Pages_Analytics')) {
            $this->analytics = new SEO_Lead_Pages_Analytics();
        }
        
        if (class_exists('SEO_Lead_Pages_Template_Manager')) {
            $this->template_manager = new SEO_Lead_Pages_Template_Manager();
        }
        
        // Initialize new premium features
        if (class_exists('GEO_Local_SEO_AI_Content_Manager')) {
            $this->ai_content_manager = new GEO_Local_SEO_AI_Content_Manager();
        }
        
        if (class_exists('GEO_Local_SEO_PageSpeed_Manager')) {
            $this->pagespeed_manager = new GEO_Local_SEO_PageSpeed_Manager();
        }
    }
    
    public function plugins_loaded() {
        // Load text domain for translations
        load_plugin_textdomain('seo-lead-pages-generator', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    public function activate() {
        // Check if WordPress is loaded properly
        if (!function_exists('wp_get_current_user')) {
            return;
        }
        
        // Create database tables
        if (class_exists('SEO_Lead_Pages_Database')) {
            try {
                $this->database = new SEO_Lead_Pages_Database();
                $this->database->create_tables();
            } catch (Exception $e) {
                error_log('GEO Local SEO: Database creation failed: ' . $e->getMessage());
            }
        }
        
        // Set default options
        try {
            add_option('seo_lead_pages_contact_email', get_option('admin_email'));
            add_option('seo_lead_pages_phone_number', '');
            add_option('seo_lead_pages_custom_templates', array());
        } catch (Exception $e) {
            error_log('GEO Local SEO: Options creation failed: ' . $e->getMessage());
        }
        
        // Flush rewrite rules
        try {
            flush_rewrite_rules();
        } catch (Exception $e) {
            error_log('GEO Local SEO: Rewrite rules flush failed: ' . $e->getMessage());
        }
    }
    
    public function deactivate() {
        // Check if WordPress is loaded properly
        if (!function_exists('wp_get_current_user')) {
            return;
        }
        
        // Flush rewrite rules
        try {
            flush_rewrite_rules();
        } catch (Exception $e) {
            error_log('GEO Local SEO: Rewrite rules flush failed during deactivation: ' . $e->getMessage());
        }
    }
}

// Initialize the plugin
new GEO_Local_SEO_Lead_Pages_Generator();

// Add some helper functions
if (!function_exists('seo_lead_pages_get_option')) {
    function seo_lead_pages_get_option($key, $default = '') {
        return get_option('seo_lead_pages_' . $key, $default);
    }
}

if (!function_exists('seo_lead_pages_update_option')) {
    function seo_lead_pages_update_option($key, $value) {
        return update_option('seo_lead_pages_' . $key, $value);
    }
}

if (!function_exists('seo_lead_pages_get_phone_number')) {
    function seo_lead_pages_get_phone_number() {
        return seo_lead_pages_get_option('phone_number', '');
    }
}

if (!function_exists('seo_lead_pages_get_contact_email')) {
    function seo_lead_pages_get_contact_email() {
        return seo_lead_pages_get_option('contact_email', get_option('admin_email'));
    }
}
